<?php

class PublicationsModel extends Model
{
    /** @var PublicationsBase */
    var $controller;

    var $langTypes = array(
        'title'        => TYPE_STR, // название
    );

    var $langItems = array(
        'title'          => TYPE_STR,  // заголовок
        'author'         => TYPE_NOTAGS,  // имя автора
        'img_text'       => TYPE_STR,  // подпись к изображению
        'content_short'  => TYPE_STR,  // краткое описание
        'content'        => TYPE_STR, // описание
        'content_search' => TYPE_STR,  // Publicator-поиск
        'title_alt'      => TYPE_NOTAGS,  // подпись для изображений
        'mtitle'         => TYPE_NOTAGS, // meta-title
        'mkeywords'      => TYPE_NOTAGS, // meta-keywords
        'mdescription'   => TYPE_NOTAGS, // meta-description
        'share_title'    => TYPE_NOTAGS, // meta-site-title
        'share_description' => TYPE_NOTAGS, // meta-share-description
        'share_sitename'    => TYPE_NOTAGS, // meta-share-sitename
    );

    var $langCategories = array(
        'title'        => TYPE_NOTAGS,  # название
        'titleh1'      => TYPE_NOTAGS,  # заголовок H1
        'mtitle'       => TYPE_NOTAGS,  # meta-title
        'mkeywords'    => TYPE_NOTAGS,  # meta-keywords
        'mdescription' => TYPE_NOTAGS,  # meta-description
    );

    /**
     * Сохранение публикации
     * @param integer $nItemID ID
     * @param array $aData данные
     * @param integer $nTypeID ID типа или 0 (текущий)
     * @return bool|int
     */
    function itemSave($nItemID, array $aData, $nTypeID = 0)
    {
        if (empty($aData)) return false;

        $aData['modified'] = $this->db->now();
        $aData['modified_uid'] = User::id();

        if($nItemID > 0)
        {
            $res = $this->db->update($this->itemsTable($nTypeID), array_diff_key($aData, $this->langItems), array('id'=>$nItemID));
            if ($res) {
                $this->db->langUpdate(array('id'=>$nItemID, 'type_id' => $this->itemsType($nTypeID)), $aData, $this->langItems, TABLE_PUB_ITEMS_LANG);
            }
            return $res;
        } else {
            $aData['created'] = $this->db->now();
            $nItemID = $this->db->insert($this->itemsTable($nTypeID), array_diff_key($aData, $this->langItems));
            if($nItemID){
                $this->db->langInsert(array('id' => $nItemID, 'type_id' => $this->itemsType($nTypeID)), $aData, $this->langItems, TABLE_PUB_ITEMS_LANG);
            }
            return $nItemID;
        }
    }

    /**
     * Данные о записи по ID
     * @param integer $nItemID ID записи
     * @param array $aFields список полей
     * @param integer $nTypeID ID типа или 0 (текущий)
     * @return mixed
     */
    public function itemData($nItemID, array $aFields = array(), $nTypeID = 0)
    {
        return $this->itemDataByFilter(array('id'=>$nItemID), $aFields, $nTypeID);
    }

    /**
     * Накручиваем счетчик просмотров
     * @param integer $nItemID ID записи
     * @param integer $nTypeID ID типа или 0 (текущий)
     */
    public function itemViewsUpdate($nItemID, $nTypeID = 0)
    {
        return $this->db->update($this->itemsTable($nTypeID), array('views = views + 1'), array('id'=>$nItemID));
    }

    /**
     * Данные о записи по фильтру
     * @param array $aFilter фильтр
     * @param array $aFields список полей
     * @param integer $nTypeID ID типа или 0 (текущий)
     * @return mixed
     */
    public function itemDataByFilter(array $aFilter = array(), array $aFields = array('*'), $nTypeID = 0)
    {
        $aFilter = $this->prepareFilter($aFilter, 'I');

        if (empty($aFields)) {
            $aFields = array('I.*');
        }

        $aData = $this->db->one_array('SELECT I.' . join(', I.', $aFields) . '
               FROM ' . $this->itemsTable($nTypeID) . ' I
               ' . $aFilter['where'] . '
               LIMIT 1', $aFilter['bind']
        );
        if (empty($aData)) $aData = array();

        return $aData;
    }

    /**
     * Список новостей компании (фронтенд)
     * @param integer $nCompanyID ID компании
     * @param bool $bCount только подсчет кол-ва объявлений
     * @param string $sqlLimit
     * @return array
     */
    function itemsListCompany($nCompanyID, $bCount = false, $sqlLimit = '') # frontend
    {
        $sqlFilter = array('I.enabled = 1');
        $typeID = Publications::TYPE_COMPANY;
        $tableItems = $this->itemsTable($typeID);

        $aBind = array();
        if ($nCompanyID > 0) {
            $sqlFilter[] = 'I.company_id = :company';
            $aBind[':company'] = $nCompanyID;
        }

        if ($bCount) {
            return $this->db->one_data('SELECT COUNT(I.id) FROM '.$tableItems.' I WHERE '.join(' AND ', $sqlFilter), $aBind);
        }
        if (is_numeric($sqlLimit)) {
            $sqlLimit = $this->db->prepareLimit(0, $sqlLimit);
        }
        $aBind[':lang'] = LNG;

        return $this->db->select('SELECT I.id, IL.title, IL.title_alt, I.link, I.publicated, I.img,
                    IL.content_short, IL.content, I.content_type, C.title as company_title
                FROM '.$tableItems.' I,
                     '.TABLE_ITEMS_LANG.' C,
                     '.TABLE_PUB_ITEMS_LANG.' IL
                WHERE '.join(' AND ', $sqlFilter).
                    $this->langAnd($typeID, true, 'I', 'IL').'
                    AND I.company_id = C.id AND C.lang = :lang
                ORDER BY I.publicated DESC
                '.$sqlLimit, $aBind);
    }

    /**
     * Список новостей компаний в категории
     * @param integer $nItemCatID ID категории компаний
     * @param bool $bCount только подсчет кол-ва
     * @param string $sqlLimit
     * @return array
     */
    function itemsListCompanyCategory($nItemCatID, $bCount = false, $sqlLimit = '') # frontend
    {
        $sqlFilter = array('I.enabled = 1');
        $typeID = Publications::TYPE_COMPANY;
        $tableItems = $this->itemsTable($typeID);

        $aBind = array();
        $sqlFilter[] = 'II.id = I.company_id';
        if ($nItemCatID > 0) {
            $sqlFilter[] = 'II.cat_id = :cat';
            $aBind[':cat'] = $nItemCatID;
        }

        if ($bCount) {
            return $this->db->one_data('SELECT COUNT(I.id) FROM '.$tableItems.' I, '.TABLE_ITEMS.' II WHERE '.join(' AND ', $sqlFilter), $aBind);
        }
        if (is_numeric($sqlLimit)) {
            $sqlLimit = $this->db->prepareLimit(0, $sqlLimit);
        }
        $aBind[':lang'] = LNG;

        return $this->db->select('SELECT I.id, IL.title, IL.title_alt, I.link, I.publicated, I.img,
                    IL.content_short, I.content_type, C.title as company_title
                FROM '.$tableItems.' I,
                     '.TABLE_ITEMS_LANG.' C,
                     '.TABLE_PUB_ITEMS_LANG.' IL,
                     '.TABLE_ITEMS.' II
                WHERE '.join(' AND ', $sqlFilter).
            $this->langAnd($typeID, true, 'I', 'IL').'
                    AND I.company_id = C.id AND C.lang = :lang
                ORDER BY I.publicated DESC
                '.$sqlLimit, $aBind);
    }

    /**
     * Перестраивание ссылок во всех записях
     */
    public function itemsLinksRebuild()
    {
        $controller = $this->controller;
        $typesData = $controller->getTypes();
        foreach ($typesData as $typeData)
        {
            $controller->type = $typeData;
            $typeID = $typeData->id;
            $this->db->select_rows_chunked($this->itemsTable($typeID), array('id','category_id','category_id2','city_id','title_params'),
                                           array(),'id',array(), function ($items) use ($typeID, $controller) {
                foreach ($items as $v) {
                    $title_params = func::unserialize($v['title_params']);
                    if (!isset($title_params['title'])) continue;
                    $v['title'] = $title_params['title'];
                    $v['link']  = $title_params['link'];
                    if ($controller->prepareItemTitle($v['id'], $v)) {
                        $controller->model->itemSave($v['id'], array(
                                'link'  => $v['link'],
                                'title' => $v['title'],
                            ), $typeID
                        );
                    }
                } unset($v);
            }, 50);
        }
    }

    /**
     * Название таблицы записей типа
     * @param int $typeID ID типа или 0 (текущий тип)
     * @return string
     */
    public function itemsTable($typeID = 0)
    {
        if (empty($typeID) || $typeID < 0) {
            return $this->controller->type->table();
        } else {
            return $this->controller->getTypeSettings($typeID)->table();
        }
    }

    /**
     * ID типа
     * @param int $typeID или 0 (ID текущего)
     * @return int
     */
    public function itemsType($typeID = 0)
    {
        if (empty($typeID) || $typeID < 0) {
            return $this->controller->type->id;
        } else {
            return $typeID;
        }
    }

    /**
     * Сохранение настроек типа публикации
     * @param integer $nTypeID ID
     * @param array $aData данные
     * @return bool|int
     */
    function typeSave($nTypeID, array $aData)
    {
        if (empty($aData)) return false;

        if($nTypeID > 0)
        {
            $this->db->langFieldsModify($aData, $this->langTypes, $aData);
            return $this->db->update(TABLE_PUB_TYPES, $aData, array('id'=>$nTypeID));
        } else {
            $this->db->langFieldsModify($aData, $this->langTypes, $aData);
            $nTypeID = $this->db->insert(TABLE_PUB_TYPES, $aData, 'id');
            if ($nTypeID) {
                $this->db->update(TABLE_PUB_TYPES, array('num'=>$nTypeID), array('id'=>$nTypeID));
            }
            return $nTypeID;
        }
    }

    /**
     * Формирование "AND связки" с языковой таблицей
     * @param integer $nTypeID ID типа
     * @param boolean $bAnd добавлять "AND"
     * @param string $sTablePrefix префикс таблицы с которой связываем
     * @param string $sLangPrefix префикс языковой таблицы
     * @param mixed $sLang keyword языка или FALSE (текущий)
     * @return string
     */
    public function langAnd($nTypeID, $bAnd = true, $sTablePrefix = 'I', $sLangPrefix = 'L', $sLang = false)
    {
        return ($bAnd ? ' AND ' : '').$sTablePrefix.'.id = '.$sLangPrefix.'.id AND '.$sLangPrefix.'.lang = '.$this->db->str2sql( ($sLang===false ? LNG : $sLang) ).' AND '.$sLangPrefix.'.type_id = '.$this->db->str2sql($nTypeID);
    }

    function getLocaleTables()
    {
        return array(
            TABLE_PUB_TYPES => array('type'=>'fields','fields'=>$this->langTypes),
            TABLE_PUB_CATEGORIES => array('type'=>'table','fields'=>$this->langCategories),
        );
    }

}