<?php

use bff\db\Publicator;
use bff\utils\Files;

# таблицы
define('TABLE_PUB_TYPES',           DB_PREFIX.'pub_types');
define('TABLE_PUB_ITEMS_',          DB_PREFIX.'pub_items_');
define('TABLE_PUB_ITEMS_LANG',      DB_PREFIX.'pub_items_lang');
define('TABLE_PUB_COMMENTS',        DB_PREFIX.'pub_comments');
define('TABLE_PUB_TAGS',            DB_PREFIX.'pub_tags');
define('TABLE_PUB_TAGS_IN',         DB_PREFIX.'pub_tags_in');
define('TABLE_PUB_CATEGORIES',      DB_PREFIX.'pub_categories');
define('TABLE_PUB_CATEGORIES_LANG', DB_PREFIX.'pub_categories_lang');
define('TABLE_PUB_CATEGORIES_IN',   DB_PREFIX.'pub_categories_in');
define('TABLE_PUB_CALENDAR',        DB_PREFIX.'pub_calendar');

/**
 * Модуль "Publications"
 * настройки в config:
 *   1) pub_items_mod_{key} - счетчик непромодерированных публикаций указанного типа публикации (добавляемые пользователями)
 *   2) pub_comments_mod_all - общий счетчик непромодерированных комментариев (всех типов публикаций, с модерацией комментариев)
 */

abstract class PublicationsBase extends Module
{
    /** @var PublicationsModel */
    public $model = null;
    protected $securityKey = '049b8e6fcd162792b8385f2d927c3995';
    
    /** @var bool компактное меню в админ панели */
    var $menuMini = true;

    # типы контента
    const contentTypePhoto = 1;
    const contentTypeVideo = 2;
    
    # типы публикаций
    const TYPE_NEWS     = 1; # новости
    const TYPE_REPORTS  = 2; # фоторепортажи
    const TYPE_ARTICLES = 3; # статьи
    const TYPE_NAROD    = 4; # народные новости
    const TYPE_VIDEO    = 5; # видео
    const TYPE_COMPANY  = 6; # новости компаний

    /** @var PublicationsSettings объект настроек текущего типа публикации */
    var $type = null;

    public function init()
    {
        parent::init();

        $this->module_title = 'Публикации';

        bff::autoloadEx(array(
            'PublicationsSettings' => array('app', 'modules/publications/publications.settings.php'),
            'PublicationsImages'   => array('app', 'modules/publications/publications.images.php'),
            'PublicationsVideo'    => array('app', 'modules/publications/publications.video.php'),
            'PublicationsTags'     => array('app', 'modules/publications/publications.tags.php'),
        ));
    }

    /**
     * Shortcut
     * @return Publications
     */
    public static function i()
    {
        return bff::module('publications');
    }

    /**
     * Shortcut
     * @return PublicationsModel
     */
    public static function model()
    {
        return bff::model('publications');
    }

    /**
     * Формирование URL
     * @param string $key ключ
     * @param mixed $opts параметры
     * @param boolean $dynamic динамическая ссылка
     * @return string
     */
    public static function url($key = '', $opts = array(), $dynamic = false)
    {
        $base = static::urlBase(LNG, $dynamic);
        switch ($key)
        {
            # Просмотр записи
            case 'view':
                if (is_string($opts)) {
                    return strtr($opts, array(
                        '{sitehost}' => (bff::isMobile() ? config::sys('site.mobile.host') : SITEHOST) . bff::locale()->getLanguageUrlPrefix(),
                    ));
                }
                break;
             # Список записей типа публикации
            case 'list':
                if (!is_array($opts)) {
                    $type = intval($opts);
                    $opts = array();
                } else {
                    $type = 0;
                    if (isset($opts['type'])) {
                        $type = intval($opts['type']);
                    }
                }
                $type = static::i()->typeKey($type);
                return $base.'/'.$type.'/' . static::urlQuery($opts, array('type'));
                break;
            # Список записей за указанную дату
            case 'date':
                return static::url('list', $opts['type']).'date'.
                    ( ! empty($opts['date']) ? date('Y-m-d', $opts['date']).'/' : '');
            # Список записей в категории
            case 'category':
                if ( ! isset($opts['type'])) break;
                return static::url('list', $opts['type'], $dynamic).
                    ( ! empty($opts['keyword']) ? $opts['keyword'].'/' : '').
                    static::urlQuery($opts, array('type', 'keyword'));
                break;
            # Форма добавления народной новости
            case 'add':
                if( ! empty($opts['keyword'])){
                    $keyword = $opts['keyword'];
                    unset($opts['keyword']);
                }
                $base .= '/'.$keyword.'/add.html'.static::urlQuery($opts);
                break;
        }
        return $base;
    }

    /**
     * Корректируем динамическую ссылку в заголовке
     * @param string $title @ref
     * @return string
     */
    public static function urlInTitle(&$title)
    {
        return strtr($title, array(
            '{sitehost}' => (bff::isMobile() ? config::sys('site.mobile.host') : SITEHOST) . bff::locale()->getLanguageUrlPrefix(),
        ));
    }

    /**
     * Описание seo шаблонов страниц
     * @return array
     */
    public function seoTemplates()
    {
        $dateMacros = array(
            'date' => array('t' => 'Дата ('.tpl::date_format2(978300000, false, true, ' ', ', ', true).')'),
            'date2' => array('t' => 'Дата ('.tpl::date_format2(978300000, false, false).')'),
            'date3' => array('t' => 'Дата ('.tpl::dateFormat(978300000).')'),
        );
        $aTemplates = array(
            'groups' => array(
                'def' => array('t'=>'Общие'),
            ),
            'pages' => array(
                'list-category' => array( // listing_category
                    't'      => 'Список в рубрике',
                    'inherit'=> true,
                    'macros' => array(
                        'type' => array('t' => 'Тип публикации'),
                        'category' => array('t' => 'Название рубрики'),
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        'titleh1' => array(
                            't'      => 'Заголовок H1',
                            'type'   => 'text',
                        ),
                    ),
                ),
                'list-date' => array( // listing_date
                    't'      => 'Список по дате',
                    'macros' => array_merge(array(
                        'region' => array('t' => 'Регион'),
                    ), $dateMacros),
                    'fields' => array(
                        // ...
                    ),
                ),
                'view' => array( // view
                    't'      => 'Просмотр публикации',
                    'inherit'=> true,
                    'macros' => array(
                        'title'  => array('t' => 'Заголовок'),
                        'description'  => array('t' => 'Описание (до 150 символов)'),
                        'tags'   => array('t' => 'Теги публикации'),
                        'type'   => array('t' => 'Тип публикации'),
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        'share_title'       => array(
                            't'    => 'Заголовок (поделиться в соц. сетях)',
                            'type' => 'text',
                        ),
                        'share_description' => array(
                            't'    => 'Описание (поделиться в соц. сетях)',
                            'type' => 'textarea',
                        ),
                        'share_sitename'    => array(
                            't'    => 'Название сайта (поделиться в соц. сетях)',
                            'type' => 'text',
                        ),
                    ),
                ),
                'view-mobile' => array( // pda_view
                    't'      => 'Просмотр публикации (mobile)',
                    'inherit'=> true,
                    'macros' => array(
                        'title'  => array('t' => 'Заголовок'),
                        'description'  => array('t' => 'Описание (до 150 символов)'),
                        'type'   => array('t' => 'Тип публикации'),
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        'share_title'       => array(
                            't'    => 'Заголовок (поделиться в соц. сетях)',
                            'type' => 'text',
                        ),
                        'share_description' => array(
                            't'    => 'Описание (поделиться в соц. сетях)',
                            'type' => 'textarea',
                        ),
                        'share_sitename'    => array(
                            't'    => 'Название сайта (поделиться в соц. сетях)',
                            'type' => 'text',
                        ),
                    ),
                ),
            ),
        );

        foreach ($this->getTypes() as $type)
        {
            $typeID = $type->id;
            $aTemplates['groups'][$typeID] = array('t'=>$type->title);
            $aTemplates['pages']['list-'.$typeID] = array( // listing_all, news_all, video_all
                't'      => 'Список (все рубрики)',
                'list'   => !in_array($typeID, array(self::TYPE_NEWS, self::TYPE_VIDEO)),
                'group'  => $typeID,
                'macros' => array(
                    'region' => array('t' => 'Регион'),
                ),
                'fields' => array(
                    'titleh1' => array(
                        't'      => 'Заголовок H1',
                        'type'   => 'text',
                    ),
                ),
            );
            if (in_array($typeID, array(self::TYPE_NEWS, self::TYPE_REPORTS, self::TYPE_ARTICLES))) {
                $aTemplates['pages']['list-mobile-'.$typeID] = array( // pda_list
                    't'      => 'Список (mobile)',
                    'group'  => $typeID,
                    'macros' => array_merge(array(
                        // ...
                    ), $dateMacros),
                    'fields' => array(
                        'titleh1' => array(
                            't'      => 'Заголовок H1',
                            'type'   => 'text',
                        ),
                    ),
                );
            }
        }

        return $aTemplates;
    }

    public static function imagesMaxSize($format = false)
    {
        $size = (int)config::sys('publications.images.maxsize', 5242880);
        if ($format) {
            $size = tpl::filesize($size);
        }
        return $size;
    }

    # ---------------------------------------------------------------------------------------------
    # Записи

    /**
     * Обработка данных записи публикации 
     * @param mixed $nItemID ID записи
     * @param array $aData текущие данные о записи
     * @return array
     */
    function processItemData($nItemID = 0, array $aData = array())
    {
        $aParams = array(              
            'link'          => TYPE_STR,   # ссылка для заголовка
            'category_id'   => TYPE_UINT,  # id категории
            'country_id'    => TYPE_UINT,  # id страны
            'city_id'       => TYPE_UINT,  # id города
            'user_id'       => TYPE_UINT,  # id пользователя
            'img'           => TYPE_STR,   # имя файла изображения
            'img_crop'      => TYPE_STR,   # настройки crop'а изображения
            'content_type'  => TYPE_UINT,  # ! тип контента
            'enabled'       => TYPE_BOOL,  # публиковать
            'main'          => TYPE_BOOL,  # на главной
            'extra'         => TYPE_ARRAY, # доп. параметры
            'mtemplate'     => TYPE_BOOL,  # использовать общий шаблон SEO
        );

        if ($this->type->contentPublicator()){
            $aParams['content'] = TYPE_ARRAY;
            unset($this->model->langItems['content']);
        }
        if ($this->type->id == self::TYPE_COMPANY) {
            $aParams['company_id'] = TYPE_UINT;
        }

        $aData = $this->input->postm($aParams);
        $this->input->postm_lang($this->model->langItems, $aData);

        $aParamsExtra = array(
            'source_link' => TYPE_ARRAY_STR,
            'xml_yandex'  => TYPE_BOOL,
        );
        switch ($this->type->id)
        {
            case 0: {
                /** 
                * add extra settings by typeID, 
                * @example: $aParamsExtra['test'] = TYPE_STR;
                */
            } break;
            case self::TYPE_VIDEO: {
                $aParamsExtra['video_top'] = TYPE_BOOL;
            } break;

        }
        
        $this->input->clean_array($aData['extra'], $aParamsExtra);
        
        if (Request::isPOST())
        {
            if ($this->type->categoriesEnabled() && $this->type->subcategoriesEnabled()){
                if( ! empty($aData['category_id'])){
                    $aCategories = $this->db->one_array('SELECT id, pid FROM '.TABLE_PUB_CATEGORIES.' WHERE id = :id', array(':id' => $aData['category_id']));
                if( ! empty($aCategories['pid'])){
                        $aData['category_id'] = $aCategories['pid'];
                        $aData['category_id2'] = $aCategories['id'];
                    }
                }
            }

            if ($nItemID) {
                $this->prepareItemTitle($nItemID, $aData);
            }

            foreach ($aData['title'] as $lng=>$t) {
                $aData['title_alt'][$lng] = HTML::escape(strip_tags(strtr($t, array('[link]'=>'','[/link]'=>''))));
            }
            if( (!$nItemID && $this->type->publicatedDateEnabled()) || # добавляем запись, дата: "при добавлении", "редактируемая"
                 ($nItemID && $this->type->publicatedDateEditable()) ) # редактируем запись, дата: "редактируемая"
            {
                $this->input->postm(array(
                    'publicate_date'   => TYPE_STR,
                    'publicate_time_h' => TYPE_UINT,
                    'publicate_time_m' => TYPE_UINT,
                ), $aData);
                
                if($aData['publicate_time_h']>23) $aData['publicate_time_h'] = 0;
                if($aData['publicate_time_m']>59) $aData['publicate_time_m'] = 0;
                $aData['publicated'] = $aData['publicate_date'] . ($this->type->publicatedTimeEnabled() ? ' '. sprintf('%02d:%02d:%02d', $aData['publicate_time_h'], $aData['publicate_time_m'], 0) : '');

                $aData['publicated'] = date('Y-m-d H:i:00', strtotime($aData['publicated'])); # форматируем для записи в БД

                unset($aData['publicate_date'], $aData['publicate_time_h'], $aData['publicate_time_m']);
            } elseif( $nItemID && $this->type->publicatedTimeEditable() ) { # редактируем запись, редактируем только время
                $this->input->postm(array(
                    'publicate_time_h' => TYPE_UINT,
                    'publicate_time_m' => TYPE_UINT,
                ), $aData);
                if($aData['publicate_time_h']>23) $aData['publicate_time_h'] = 0;
                if($aData['publicate_time_m']>59) $aData['publicate_time_m'] = 0; 
                $aData['publicate_time'] = sprintf('%02d:%02d:00', $aData['publicate_time_h'], $aData['publicate_time_m']);
                unset($aData['publicate_time_h'], $aData['publicate_time_m']);
            }
            
            if ($nItemID>0 && $this->type->tagsEnabled()) {
                $this->initTags($this->type->id)->tagsSave($nItemID);
            }
            
            foreach($aData['extra'] as $k=>$v) {
                $aData['extra_'.$k] = $v;
            } unset($aData['extra']);

            if($this->type->extra['source_link']) {
                $this->input->clean_array($aData['extra_source_link'], array(
                    'link'  => TYPE_STR,
                    'title' => TYPE_STR,
                ));
                $aData['extra_source_link'] = serialize( $aData['extra_source_link'] );
            } else {
                unset( $aData['extra_source_link'] );
            }
            
            if( ! $this->type->extra['main']) {
                unset($aData['main']);
            }
            if( $this->type->contentPublicator() ) {
                $aItemData = array('id' => $nItemID);
                if( ! empty($aData['publicated'])) {
                    $aItemData['publicated'] = $aData['publicated'];
                }
                $aDataPublicator = $this->initPublicator($aItemData)->dataPrepare($aData['content'], $nItemID);
                $aData['content'] = $aDataPublicator['content'];
                $aData['content_search'] = $aDataPublicator['content_search'];
                $aData['content_type'] = $this->updateContentType($aDataPublicator);
                $aData['photos'] = sizeof($aDataPublicator['photos']);
            }
            # на случай если настройки типа были изменены в процессе,
            # оставляем неиспользуемые данные БЕЗ изменений
            $unsetData = array(
                //'regions' => array('country_id', 'city_id'),
                'content_short' => array('content_short'),
                'categories' => array('category_id'),
                'img' => array('img','img_text'),
                'author' => array('user_id','author'),
            );
            foreach ($unsetData as $k=>$v) {
                if (!$this->type->$k) {
                    if (is_string($v)) $v = array($v);
                    foreach ($v as $v2) {
                        unset($aData[$v2]);
                    }
                }
            }
            if ($this->type->id == self::TYPE_COMPANY && $aData['company_id'] > 0) {
                $aCompanyData = Items::model()->companyData($aData['company_id']);
                if ( ! empty($aCompanyData['city_id'])) {
                    $aData['country_id'] = $aCompanyData['country_id'];
                    $aData['city_id'] = $aCompanyData['city_id'];
                }
            }
        } else {
            if (!$nItemID) {
                $aData['mtemplate'] = 1;
                $aData['publicated'] = date('Y-m-d H:i:00');
                foreach ($aData['extra'] as $k=>$v) {
                    $aData['extra_'.$k] = $v;
                } unset($aData['extra']);

                if ($this->type->extra['source_link']) {
                    $this->input->clean_array($aData['extra_source_link'], array(
                        'link' => TYPE_STR,
                        'title' => TYPE_STR,
                    ));
                }
                
                if ($this->type->authorUser() && !isset($aData['user_login'])) {
                    $aData['user_login'] = '';
                }
            }
        }
        
        return $aData;
    }
    
    /**
     * Инициализация объекта работы с изображением публикации
     * @param array $aItemData параметры записи array(id)
     * @param PublicationsSettings $oTypeSettings объект настроек типа публикации
     * @return PublicationsImages объект
     */
    function initImages(array $aItemData, PublicationsSettings $oTypeSettings = null)
    {
        static $cache;
        if($oTypeSettings === null) {
            $oTypeSettings = $this->type;
        }
        
        if(!empty($oTypeSettings) && $oTypeSettings->imgEnabled())
        {
            if(isset($cache)) {
                return $cache;
            }

            return ($cache = new PublicationsImages($oTypeSettings, $this->getItemsTable($oTypeSettings->keyword), $aItemData));
        }
        return false;
    }

    /**
     * Инициализация компонента PublicationsItemComments
     * @param integer $nTypeID тип
     * @return PublicationsItemComments component
     */
    public function itemComments($nTypeID = 0)
    {
        static $c;
        if(!isset($c)) {
            include_once $this->module_dir.'publications.item.comments.php';
            $c = new PublicationsItemComments();
            if( ! $nTypeID){
                $nTypeID = $this->type->id;
            }
            $c->setGroupID($nTypeID);
        }
        return $c;
    }
    
    /**
     * Формирование имени таблицы записей типа
     * @param string|integer $type keyword типа или ID
     * @return string название таблицы
     */
    public function getItemsTable($type)
    {
        if (is_numeric($type)) {
            $type = $this->typeKey($type);
        }
        return TABLE_PUB_ITEMS_.$type;
    }

    public function typeKey($typeID)
    {
        static $keys = array(
            self::TYPE_NEWS     => 'news',
            self::TYPE_REPORTS  => 'reports',
            self::TYPE_ARTICLES => 'articles',
            self::TYPE_NAROD    => 'narod',
            self::TYPE_VIDEO    => 'video',
            self::TYPE_COMPANY  => 'company',
        );
        if (isset($keys[$typeID])) {
            return $keys[$typeID];
        }
        return '';
    }
    
    /**
     * Строим путь к изображениям 
     * @param boolean $bURL стоим URL
     * @param mixed $aItemData данные о записи, array(id, size, filename, publicated)
     * @param PublicationsSettings $oTypeSettings объект настроек типа публикации
     * @return string
     */
    function getImagesPath($bURL = false, $aItemData = false, PublicationsSettings $oTypeSettings = null)
    {
        $bTemp = empty($aItemData['id']);
        if (empty($oTypeSettings)) {
            $oTypeSettings = $this->type;
        }

        if($bTemp) {
            $sPath = ( $bURL ? bff::url('tmp/pub', 'images') : bff::path('tmp/pub', 'images') );
        } else {
            $sPath = ($bURL ? bff::url($oTypeSettings->keyword, 'images') : bff::path($oTypeSettings->keyword, 'images') );
                    
            # строим путь на основе даты публикации 
            # 2006-04-05 01:50:00 => /2006/04/05/
            if(!empty($aItemData['publicated'])) 
            {
                $date = func::parse_datetime($aItemData['publicated']);
                $date_parts = array($date['year'], $date['month'], $date['day']);
                $sPathDate = join('/', $date_parts).'/';
                if(!$bURL && !is_dir( $sPath.$sPathDate )) {
                    $sPathParts = $sPath;
                    foreach($date_parts as $part) {
                        $sPathParts .= $part.'/';
                        if(!is_dir($sPathParts)) {
                            mkdir($sPathParts);
                        }
                    }
                }
                $sPath .= $sPathDate;
            }
        }
        
        if (empty($aItemData['size'])) return $sPath;
        
        $sFilePrefix = ( $bTemp ? $oTypeSettings->id.'_' : $aItemData['id'].'_' ); 
        return $sPath . $sFilePrefix . $aItemData['size'] . $aItemData['filename'];
    }

    /**
     * Формируем заголовок записи, содержащий ссылку.
     * @param integer $nItemID ID записи
     * @param array $aItemData @ref данные о записи:
     *    title - оригинальное название, с тегами [link][/link]
     *    link - ссылка
     *    [keyword] - keyword записи
     *    category|categоry_id - keyword|ID категории записи
     *    city_id - ID города
     * @param array $aLinkAttr аттрибуты ссылки
     * @return boolean
     */
    public function prepareItemTitle($nItemID, array &$aItemData, $aLinkAttr = array(), $type = null)
    {
        if (!$nItemID || empty($aItemData['title'])) {
            return false;
        }

        if( ! empty($type)) {
            $this->type = $type;
        }

        # проверка данных
        if (is_string($aItemData['title'])) {
            $title = $aItemData['title']; $aItemData['title'] = array();
            foreach ($this->locale->getLanguages() as $lng) {
                $aItemData['title'][$lng] = $title;
            }
        }
        if (!isset($aItemData['link'])) {
            $aItemData['link'] = '';
        }

        # форматируем заголовок
        foreach ($aItemData['title'] as $lng=>$title) {
            $aItemData['title'][$lng] = strtr($title, array("\r\n"=>' ',"\n"=>''));
        }

        # сохраняем для редактирования
        $aLinkAttr = HTML::attributes($aLinkAttr);
        $aItemData['title_params'] = serialize(array('title'=>$aItemData['title'], 'link'=>$aItemData['link']));
        $url = $this->itemURL($nItemID, $aItemData);

        # заголовок является полностью ссылкой (без выделения области ссылки)
        if (!$this->type->extra['title_link']) {
            foreach ($aItemData['title'] as $lng=>$title) {
                $title = strtr($title, array('[link]'=>'','[/link]'=>''));
                $title = '<a href="'.$url.'"'.$aLinkAttr.'>'.$title.'</a>';
                $aItemData['title'][$lng] = $title;
            }
            $aItemData['link'] = $url;
            return true;
        }

        # формируем заголовок с выделением области ссылки
        $link = $aItemData['link'];

        # включена настройка "задавать собственную ссылку"
        if ($this->type->extra['title_link_custom']) {
            if (empty($link) || $link === '#') {
                $link = $url;
            }
        } else {
            $link = $url;
        }

        # title => ссылка
        foreach ($aItemData['title'] as $lng=>$title) {
            if (mb_stripos($title, '[link]') !== false) {
                $title = preg_replace('|\[link\]([^\[]+)\[/link\]|iu', '<a href="'.$link.'"'.$aLinkAttr.'>$1</a>', $title);
            } else {
                # весь заголовок является ссылкой
                $title = '<a href="'.$link.'"'.$aLinkAttr.'>'.$title.'</a>';
            }
            $aItemData['title'][$lng] = $title;
        } unset($v);
        $aItemData['link'] = $link;

        return true;
    }

    /**
     * Формируем динамический URL для записи
     * @param integer $nItemID ID записи
     * @param array $aItemData данные о записи (keyword, title, category_id, category_id2, city_id)
     * @return string
     */
    protected function itemURL($nItemID, array $aItemData)
    {
        # подменяем макросы:
        # {id}
        $aReplace = array('{id}'=>$nItemID);
        # {keyword}
        if ( ! empty($aItemData['keyword'])) {
            $aReplace['{keyword}'] = $aItemData['keyword'];
        } else {
            if ( ! empty($aItemData['title'][LNG])) {
                $sKeyword = strtr($aItemData['title'][LNG], array('[link]'=>'','[/link]'=>''));
                $sKeyword = mb_strtolower(func::translit($sKeyword));
                if ( ! empty($sKeyword)) {
                    $aReplace['{keyword}'] = $sKeyword;
                }
            }
        }
        # {cat}
        if ($this->type->categoriesEnabled() && mb_stripos($this->type->linkout, '{cat}') !== false) {
            $aReplace['{cat}'] = '';
            $aNeedCats = array($aItemData['category_id']);
            if ($this->type->subcategoriesEnabled() && ! empty($aItemData['category_id2'])) {
                $aNeedCats[] = $aItemData['category_id2'];
            }
            $aCats = $this->db->select_rows_key(TABLE_PUB_CATEGORIES, 'id', array('id','pid','keyword'), array('id'=>$aNeedCats));
            if ( ! empty($aCats[ $aItemData['category_id'] ]['keyword'])) {
                $aReplace['{cat}'] = $aCats[ $aItemData['category_id'] ]['keyword'];
            }
            if ( ! empty($aItemData['category_id2']) && ! empty($aCats[ $aItemData['category_id2'] ]['keyword'])) {
                $aReplace['{cat}'] .= '/'.$aCats[ $aItemData['category_id2'] ]['keyword'];
            }
        }
        $linkout = strtr($this->type->linkout, $aReplace);

        # формируем URL
        return static::urlBase(LNG, true, array('city'=>$aItemData['city_id'])).'/'.ltrim($linkout, '/');
    }
    
    function deleteItem($nItemID, $nTypeID)
    {
        if( ! $nItemID || ! $nTypeID) return false;
        
        $type = $this->getTypeSettings($nTypeID);

        $aData = $this->model->itemData($nItemID, array('*'), $nTypeID);
        if (empty($aData) || !$nTypeID) return false;

        if( ! $aData['moderated']){
            $this->itemsModerationCounter($nTypeID, -1);
        }

        # удаляем изображения загруженные через публикатор
        if($type->contentPublicator()) {
            $oPublicator = $this->initPublicator( array('id'=>$nItemID, 'publicated'=>$aData['publicated']) );
            $oPublicator->dataDelete($aData['content'], $nItemID);
        }
        
        # удаляем изображение
        if($type->imgEnabled())
        {
            $oImages = $this->initImages( array('id'=>$nItemID, 'publicated'=>$aData['publicated']) );
            if(!empty($aData['img']) && $oImages!==false) {
                $oImages->deleteFiles( $aData['img'] );
            }
        }
        
        # удаляем комментарии
        if($type->commentsManage()) {
            $this->db->delete(TABLE_PUB_COMMENTS, array('type_id'=>$nTypeID, 'item_id'=>$nItemID));
        }
        
        # актуализируем
        if($aData['enabled']) {
            # фиксируем в календаре
            $this->calendarUpdate($nTypeID, $aData['publicated'], $aData['city_id'], -1);
            # обновляем RSS
            $this->rssUpdate($nTypeID, $aData['category_id']);
        }
        
        
        # удаляем привязку к тегам
        $this->db->delete(TABLE_PUB_TAGS_IN, array('type_id'=>$nTypeID, 'item_id'=>$nItemID));

        # удаляем мультиязычные данные
        $this->db->delete(TABLE_PUB_ITEMS_LANG, array('type_id'=>$nTypeID, 'id'=>$nItemID));

        # удаляем запись
        $res = $this->db->delete($type->table(), array('id'=>$nItemID));
        
        if( ! empty($res)) {
            if($nTypeID == self::TYPE_NAROD) {
                $this->security->userCounter('narod', -1, $aData['user_id']);
            }
            if( $type->moderationEnabled() ) {
                $this->itemsModerationCounter($nTypeID, -1);
            }
            return true;
        }
        return false;
    }
    
    function itemsModerationCounter($sKey, $nIncrement = false)
    {
        if (is_numeric($sKey)) {
            $sKey = $this->typeKey($sKey);
        }
        $sCounterKey = 'pub_items_mod_'.$sKey;
        if($nIncrement!==false) {
            config::saveCount($sCounterKey, $nIncrement, true);
        } else {
            return (int)config::get($sCounterKey, 0);
        }
    }
    
    /**
     * Инициализация компонента Publicator
     * @param array $aItemData
     * @param PublicationsSettings $oTypeSettings объект настроек типа публикации
     * @return PubPublicator объект
     */
    function initPublicator(array $aItemData, PublicationsSettings $oTypeSettings = null)
    {
        if($oTypeSettings === null) {
            $oTypeSettings = $this->type;
        }
        if (empty($oTypeSettings->content_pub['sett'])) {
            $this->errors->set(_t('pub', 'Ошибка инициализация компонента Publicator, проверьте корректность настроек типа публикации'));
            return;
        }
        $aSettings = array();
        $aSettings['images_path'] = $this->getImagesPath(false, $aItemData, $oTypeSettings);
        $aSettings['images_url']  = $this->getImagesPath(true, $aItemData, $oTypeSettings);
        # переопределяем путь к временной директории
        if( empty($aItemData['id']) ) {
            $aSettings['images_path_tmp'] = $aSettings['images_path'];
            $aSettings['images_url_tmp']  = $aSettings['images_url'];
        } else {
            $aItemData = array('id'=>0);
            $aSettings['images_path_tmp'] = $this->getImagesPath(false, $aItemData, $oTypeSettings);
            $aSettings['images_url_tmp']  = $this->getImagesPath(true, $aItemData, $oTypeSettings);
        }
        $aSettings['langs'] = $this->locale->getLanguages(true);

        static $publicator;
        if( ! isset($publicator) ) {
            include_once('publications.publicator.php');

            if( ! empty($oTypeSettings->content_pub['sett']['gallery_sz_view']) &&
                  empty($oTypeSettings->content_pub['sett']['gallery_sz_view']['watermark'])){
                if( ! bff::adminPanel()) {
                    # глобальные настройки водяного знака
                    $watermark = Site::i()->watermarkSettings();
                    if (!empty($watermark['file']['path'])) {
                        $watermarkSettings = array(
                            'watermark' => true,
                            'watermark_src' => $watermark['file']['path'],
                            'watermark_pos_x' => $watermark['pos_x'],
                            'watermark_pos_y' => $watermark['pos_y'],
                        );
                        $oTypeSettings->content_pub['sett']['gallery_sz_view'] += $watermarkSettings;
                    }
                }
            }

            $publicator = new PubPublicator($this->module_name, $oTypeSettings->content_pub['sett'], false);
            $this->attachComponent('publicator', $publicator);
        }

        if (bff::adminPanel()) {
            $aSettings['wysiwyg_scripts'] = true;
            $aSettings['wysiwyg_iframes'] = true;

        }

        $publicator->setSettings($aSettings);
        # выполняем проверку запроса только после полной инициализации настроек публикатора
        $publicator->checkRequest();

        return $publicator;
    }
    
    function updateContentType($aPublicatorData)
    {
        $nContentTypes = 0;
        if($this->type->contentTypesEnabled() && 
           ! empty($aPublicatorData['content_types']) &&
           array_sum($aPublicatorData['content_types'])>0) 
        {
            foreach($aPublicatorData['content_types'] as $type=>$typeBlocks) {
                if(!$typeBlocks) continue;
                if($type == Publicator::blockTypePhoto || $type == Publicator::blockTypeGallery) {
                    $nContentTypes |= self::contentTypePhoto;
                } else if($type == Publicator::blockTypeVideo) {
                    $nContentTypes |= self::contentTypeVideo;
                }
            }
        }
        return $nContentTypes;
    }
    
    /**
     * Ставим хук на вызов неизвестного метода, для работы с данными типа публикации:
     * @param string $sName [component|module]_[method] или [type-keyword]:[method]
     * @param array $aArgs
     * @return mixed
     */
    public function __call($sName, $aArgs = array()) 
    {                 
        if (strpos($sName, ':') !== false)
        {
            list($sTypeKey, $sMethod) = explode(':', $sName, 2);
            $aTypes = $this->getTypes();
            if ( ! empty($aTypes[$sTypeKey]) && method_exists($this, $sMethod))
            {
                $this->type = $aTypes[$sTypeKey]; # инициализируем настройки текущего типа
                $_GET['type'] = $_POST['type'] = $this->type->id;
                if ( ! bff::adminPanel()) {
                    $this->typeTemplatesDir = $this->module_dir_tpl . DS . $sTypeKey;
                    $this->itemsTable = $this->getItemsTable($sTypeKey);
                }
                if ($this->type->contentPublicator()) {
                    unset($this->model->langItems['content']);
                }
                return $this->$sMethod($this->type->id, $sTypeKey);
            }
        }

        return parent::__call($sName, $aArgs); 
    }
    
    /**
     * Построение меню админ панели для работы с модулем публикаций
     * @param CMenu $menu
     */
    function buildAdminMenu(CMenu $menu)
    {
        $aTypes = $this->getTypes(); 
        $i = 20;
        $sModule = 'publications';

        $access_Comments = $this->haveAccessTo('comments');
        if($this->menuMini)
        {
            foreach($aTypes as $sKey=>$v) {
                if(!$this->haveAccessTo('items-listing', 'publications-'.$sKey)) continue;

                $params = array('rlink'=>array('event'=>$sKey.':add'));
                if($v->moderationEnabled()) {
                    $params['counter'] = 'pub_items_mod_'.$sKey;
                }
                $menu->assign('Публикации', $v->title, $sModule, $sKey.':listing', true, $i++, $params);
                $menu->assign('Публикации', $v->title.' / Добавление', $sModule, $sKey.':add', false, $i++);
                $menu->assign('Публикации', $v->title.' / Редактирование', $sModule, $sKey.':edit', false, $i++);
            }
            
            if($access_Comments) 
            {
                $aCommentsParams = array();
                if(bff::$event!=='comments_mod') {
                    $aCommentsParams['counter'] = 'pub_comments_mod_all';
                }                
                $menu->assign('Публикации', 'Комментарии', $sModule, 'comments_mod', true, $i++, $aCommentsParams);
            }
            if($this->haveAccessTo('cats-listing'))
                $menu->assign('Публикации', 'Разделы', $sModule, 'categories', true, $i++);
            if($this->haveAccessTo('tags-listing'))
                $menu->assign('Публикации', 'Теги', $sModule, 'tags', true, $i++);
            
        } else {
            foreach($aTypes as $sKey=>$v)
            {
                if(!$this->haveAccessTo('items-listing', 'publications-'.$sKey)) continue;
                
                $menu->assign($v->title, 'Список', $sModule, $sKey.':listing', true, $i++, array('rlink'=>array('event'=>$sKey.':add')));
                $menu->assign($v->title, 'Добавление', $sModule, $sKey.':add', false, $i++);
                $menu->assign($v->title, 'Редактирование', $sModule, $sKey.':edit', false, $i++);
                if( $v->categoriesEnabled() && $this->haveAccessTo('cats-listing', (!$v->categoriesCommonOnly() ? 'publications-'.$sKey:null)) ) {
                    $menu->assign($v->title, 'Разделы', $sModule, $sKey.':categories', true, $i++);
                }
                if( $v->tagsEnabled() && $this->haveAccessTo('tags-listing',($v->tagsCustom() ? 'publications-'.$sKey : null)) ) {
                    $menu->assign($v->title, 'Теги', $sModule, $sKey.':tags', true, $i++);
                }
                if( $access_Comments && $v->commentsManage() ) {
                    $menu->assign($v->title, 'Комментарии', $sModule, $sKey.':comments_mod', true, $i++);
                }
            }
        }

        if(FORDEV)
        {
            $menu->assign('Публикации', 'Настройки типов', $sModule, 'types_listing', true, $i++, array(
                'rlink'=>array('event'=>'types_add')
            ));
            $menu->assign('Публикации', 'Создание типа', $sModule, 'types_add', false, $i++);
            $menu->assign('Публикации', 'Редактирование типа', $sModule, 'types_edit', false, $i++);
        }
    }
    
    /**
     * Обработка данных типа публикации 
     * @param array $aData @ref настройки типа публикации
     * @param mixed $nTypeID ID типа
     * @return void
     */
    function processTypeSettings(&$aData, $nTypeID = 0)
    {
        $aParams = array(              
            'keyword' => TYPE_NOTAGS,
            'regions' => TYPE_UINT,
            'regions_country' => TYPE_UINT,
            'categories'      => TYPE_UINT,
            'subcategories'   => TYPE_BOOL,
            'tags'            => TYPE_UINT,
            'comments'        => TYPE_UINT, 
            'commentsp'       => TYPE_ARRAY,
            'publicate_date'  => TYPE_UINT, # 0,1,2
            'publicate_time'  => TYPE_UINT, # 0,1
            'author'        => TYPE_UINT,
            'img'           => TYPE_BOOL,
            'imgp'          => TYPE_ARRAY,
            'content_short' => TYPE_BOOL,
            'content'       => TYPE_UINT,
            'content_pub'   => TYPE_ARRAY,
            'content_types' => TYPE_UINT,   
            'users_add'     => TYPE_UINT, 
            'users_groups'  => TYPE_ARRAY_UINT,
            'extra'         => TYPE_ARRAY,
            'linkout'       => TYPE_STR,
            'enabled'       => TYPE_BOOL,
        );
        $this->input->postm($aParams, $aData);
        $this->input->postm_lang($this->model->langTypes, $aData );

        $aData['imgp']        = PublicationsSettings::cleanArray($aData['imgp'], 'imgp');
        $aData['commentsp']   = PublicationsSettings::cleanArray($aData['commentsp'], 'commentsp');
        $aData['content_pub'] = PublicationsSettings::cleanArray($aData['content_pub'], 'content_pub');
        $aData['extra']       = PublicationsSettings::cleanArray($aData['extra'], 'extra');
        
        if (Request::isPOST())
        {
            $aData['keyword'] = $this->getKeyword($aData['keyword'], $aData['title'][LNG], TABLE_PUB_TYPES, $nTypeID);
            
            if(!array_key_exists($aData['publicate_date'], $this->getTypeSettingsForm('publicate_date', 0, false))) {
                $aData['publicate_date'] = 0;
            }
            
            //изображение 
            {
                $imgp =& $aData['imgp'];
                
                if($imgp['orig'] && $imgp['size_orig']=='') {
                    $imgp['orig'] = false;
                }
                
                # размеры изображений
                $imgSizes = array(); $imgPrefixes = array(); $cropSizes = array(); $wmSizes = array();
                if(!empty($imgp['sizes']))
                {
                    foreach($imgp['sizes'] as $v)
                    {
                        $this->input->clean_array($v, array(
                            'p' => TYPE_STR,     # префикс
                            'w' => TYPE_UINT,    # ширина
                            'h' => TYPE_UINT,    # высота
                            'wm' => TYPE_BOOL,   # watermark
                            'crop' => TYPE_BOOL, # crop
                            'del' => TYPE_BOOL,  # пометка на удаление размера
                        ));
                        
                        $prefix = mb_strtolower($v['p']);
                        $prefix = preg_replace('/[^a-z]/','', $prefix);
                        if((empty($v['p']) || $v['p']==$imgp['size_orig'] || in_array($prefix, $imgPrefixes)) || (!$v['w'] && !$v['h'])) continue;
                        
                        if($nTypeID>0 && $v['del']) {
                            # удаление размера
                            continue;
                        } else {
                            unset($v['del']);
                        }
                        if(!$v['w']) $v['w'] = false;
                        if(!$v['h']) $v['h'] = false;
                        if($v['crop']) $cropSizes[] = $prefix; 
                        if($v['wm']) $wmSizes[] = $prefix;
                        $imgPrefixes[] = $prefix;
                        
                        $v['p'] = $prefix;
                        $imgSizes[$prefix] = $v; 
                    }
                }
                $imgp['sizes'] = $imgSizes;  
                
                if(!array_key_exists($imgp['size_adm'], $imgSizes)) {
                     $imgp['size_adm'] = key($imgSizes);
                }
                
                # watermark
                $wmValid = false;
                do {
                    if(!$imgp['wm'] || !in_array($imgp['wm'], array(1,2)) || empty($imgSizes)) break;
                    
                    $wmError = false;
                    if($imgp['wm'] == 1) { # изображение
                        if(!file_exists(PATH_PUBLIC . $imgp['wm_img'])) {
                            $this->errors->set( _t('pub', 'Путь watermark-изображения "[path]" указан некорректно', array('path'=>$imgp['wm_img'])), true );
                            $wmError = true;
                        }
                        if (empty($imgp['wm_pos_x']) || !in_array($imgp['wm_pos_x'], array('left','center','right')))
                            $imgp['wm_pos_x'] = 'right';
                        if (empty($imgp['wm_pos_y']) || !in_array($imgp['wm_pos_y'], array('top','center','bottom')))
                            $imgp['wm_pos_y'] = 'bottom';                            
                    } else if($imgp['wm'] == 2) { # текст
                        if($imgp['wm_text'] == '') {
                            $this->errors->set( _t('pub', 'Текст для watermark указан некорректно') );
                            $wmError = true;
                        }
                        if (empty($imgp['wm_pos_text_x']) || !in_array($imgp['wm_pos_text_x'], array('left','center','right')))
                            $imgp['wm_pos_text_x'] = 'right';
                        if (empty($imgp['wm_pos_text_y']) || !in_array($imgp['wm_pos_text_y'], array('top','center','bottom')))
                            $imgp['wm_pos_text_y'] = 'bottom';
                        
                        $imgp['wm_pos_x'] = $imgp['wm_pos_text_x'];
                        $imgp['wm_pos_y'] = $imgp['wm_pos_text_y'];
                    }
                    
                    if($wmError || empty($wmSizes)) {
                        break; # размеры для wm не указаны
                    }
                    
                    $wmValid = true;
                } while(false);
                $imgp['wm'] = ($wmValid ? 1 : 0);
                                
                # crop
                $imgp['crop'] = ( $imgp['orig'] && !empty($imgSizes) && !empty($cropSizes) ? $cropSizes : false );
            }
            
            # publicator
            {
                $pub =& $aData['content_pub'];

                $pub['title'] = (bool)$pub['title'];
                $pub['use_wysiwyg'] = (bool)$pub['use_wysiwyg'];
                $pub['photo_wysiwyg'] = (bool)$pub['photo_wysiwyg'];

                $sett = array();
                # настройки сразу подходящие публикатору:
                foreach(array('title','use_wysiwyg','images_original',
                        'photo_wysiwyg','gallery_photos_limit',
                        'video_width','video_height') as $k) {
                    $sett[$k] = $pub[$k];
                }
                $sett['controls'] = array(Publicator::blockTypeText);

                # приводим к формату, понимаемому публикатором:
                # Photo
                if( $pub['photo_enabled'] )
                {
                    $sett['photo_sz_view'] = array(
                        'width' => $pub['photo_view_width'],
                        'height' => $pub['photo_view_height'],
                    );
                    do {
                        if( ! $pub['photo_view_wm']) break;

                        if( ! file_exists(PATH_PUBLIC . $pub['photo_view_wm_src'])) {
                            $this->errors->set( _t('', 'Путь watermark-изображения "[path]" для Фото публикатора указан некорректно', array('path'=>$pub['photo_view_wm_src'])), true );
                        } else {
                            $sett['photo_sz_view'] = array_merge($sett['photo_sz_view'], array(
                                'watermark'=>true, 'watermark_src'=>PATH_PUBLIC . $pub['photo_view_wm_src'], 'watermark_resizable'=>false,
                                'watermark_pos_x'=>$pub['photo_view_wm_pos_x'], 'watermark_pos_y'=>$pub['photo_view_wm_pos_y']));
                        }
                    } while(false);
                    $sett['controls'][] = Publicator::blockTypePhoto;
                }
                # Gallery
                if( $pub['gallery_enabled'] )
                {
                    $sett['gallery_sz_view'] = array(
                        'width' => $pub['gallery_view_width'],
                        'height' => $pub['gallery_view_height'],
                    );
                    do {
                        if( ! $pub['gallery_view_wm']) break;

                        if( ! file_exists(PATH_PUBLIC . $pub['gallery_view_wm_src'])) {
                            $this->errors->set( _t('', 'Путь watermark-изображения "[path]" для Фотогалереи публикатора указан некорректно', array('path'=>$pub['gallery_view_wm_src'])), true );
                        } else {
                            $sett['gallery_sz_view'] = array_merge($sett['gallery_sz_view'], array(
                                'watermark'=>true, 'watermark_src'=>PATH_PUBLIC . $pub['gallery_view_wm_src'], 'watermark_resizable'=>false,
                                'watermark_pos_x'=>$pub['gallery_view_wm_pos_x'], 'watermark_pos_y'=>$pub['gallery_view_wm_pos_y']));
                        }
                    } while(false);
                    $sett['controls'][] = Publicator::blockTypeGallery;
                }
                # Video
                if( $pub['video_enabled'] )
                {
                    $sett['controls'][] = Publicator::blockTypeVideo;
                }
                
                $pub['sett'] = $sett;
            }
            
            # "города" => но, не указана страна
            if($aData['regions'] == 1 && empty($aData['regions_country'])) 
                $aData['regions'] = 0;
                
            if($aData['users_add'])
            {
                $aGroups = $aData['users_groups'];
                $aFrontendGroups = Users::model()->groups(null, true);
                $aFrontendGroups = func::array_transparent($aFrontendGroups, 'group_id', true);
                foreach($aGroups as $k=>$v) {
                    if(!array_key_exists($v, $aFrontendGroups))
                        unset($aGroups[$k]);
                }
                $aData['users_groups'] = $aGroups;
            }
            
            if (empty($aData['linkout'])) {
                $aData['linkout'] = "/{$aData['keyword']}/{id}.html";
            }
        }
    }
    
    function getTypeSettingsForm($sKey, $mCurrent, $bHTML = true)
    {
        $sHTML = '';
        switch ($sKey)
        {
            case 'regions': 
            {
                $aValues = array(0=>'нет', 1=>'города', 2=>'страны и города');
                if($bHTML) {
                    foreach($aValues as $k=>$v) {
                        $sHTML .= '<option value="'.$k.'"'.($mCurrent==$k?' selected':'').'>'.$v.'</option>';
                    }
                } else {
                    return $aValues;
                }
            } break;              
            case 'regions_country': 
            {
                $aCountries = Geo::countryList();
                if($bHTML) {
                    $sHTML = HTML::selectOptions($aCountries, $mCurrent, '- выбрать страну -', 'id', 'title');
                } else {
                    return $aCountries;
                }
            } break;
            case 'categories':
            {
                $aValues = array(0=>'нет', 1=>'свои', 2=>'свои + общие (выборочное наследование)', 3=>'только общие (полное наследование)');
                if($bHTML) {
                    foreach($aValues as $k=>$v) {
                        $sHTML .= '<option value="'.$k.'"'.($mCurrent==$k?' selected':'').'>'.$v.'</option>';
                    }
                } else {
                    return $aValues;
                }
            } break;
            case 'tags':
            {
                $aValues = array(0=>'нет', 1=>'свои', 2=>'общие');
                if($bHTML) {
                    foreach($aValues as $k=>$v) {
                        $sHTML .= '<option value="'.$k.'"'.($mCurrent==$k?' selected':'').'>'.$v.'</option>';
                    }
                } else {
                    return $aValues;
                }
            } break;  
            case 'comments':
            {
                $aValues = array(0=>'нет', 1=>'древовидные', 2=>'недревовидные', /*3=>'facebook', 4=>'vkontakte'*/);
                if($bHTML) {
                    foreach($aValues as $k=>$v) {
                        $sHTML .= '<option value="'.$k.'"'.($mCurrent==$k?' selected':'').'>'.$v.'</option>';
                    }
                } else {
                    return $aValues;
                }
            } break;  
            case 'comments_mod':
            {
                $aValues = array(1=>'премодерация', 0=>'постмодерация');
                if($bHTML) {
                    foreach($aValues as $k=>$v) {
                        $sHTML .= '<option value="'.$k.'"'.($mCurrent==$k?' selected':'').'>'.$v.'</option>';
                    }
                } else {
                    return $aValues;
                }
            } break;
            case 'publicate_date':
            {
                $aValues = array(0=>'нет', 1=>'при добавлении', 2=>'редактируемая');
                if($bHTML) {
                    foreach($aValues as $k=>$v) {
                        $sHTML .= '<option value="'.$k.'"'.($mCurrent==$k?' selected':'').'>'.$v.'</option>';
                    }
                } else {
                    return $aValues;
                }
            } break;               
            case 'publicate_time':
            {
                $aValues = array(0=>'нет', 1=>'есть', 2=>'есть, редактируемое');
                if($bHTML) {
                    foreach($aValues as $k=>$v) {
                        $sHTML .= '<option value="'.$k.'"'.($mCurrent==$k?' selected':'').'>'.$v.'</option>';
                    }
                } else {
                    return $aValues;
                }
            } break;
            case 'author':
            {
                $aValues = array(0=>'нет', 1=>'текст', 2=>'пользователь');
                if($bHTML) {
                    foreach($aValues as $k=>$v) {
                        $sHTML .= '<option value="'.$k.'"'.($mCurrent==$k?' selected':'').'>'.$v.'</option>';
                    }
                } else {
                    return $aValues;
                }
            } break;
            case 'users_add':
            {
                $aValues = array(0=>'нет', 1=>'премодерация', 2=>'постмодерация', 3=>'без модерации');
                if($bHTML) {
                    foreach($aValues as $k=>$v) {
                        $sHTML .= '<option value="'.$k.'"'.($mCurrent==$k?' selected':'').'>'.$v.'</option>';
                    }
                } else {
                    return $aValues;
                }
            } break;
            case 'users_groups': 
            {
                $aFrontendGroups = Users::model()->groups(null, true);
                if($bHTML) {
                    foreach($aFrontendGroups as $v) {
                        $sHTML .= '<option value="'.$v['group_id'].'" style="color:'.$v['color'].';" '.(in_array($v['group_id'], $mCurrent)?' selected':'').'>'.$v['title'].'</option>';
                    }
                } else {
                    return $aFrontendGroups;
                }
            } break;
        }
        
        return $sHTML;
    }

    /**
     * Получаем настройки всех типов публикаций
     * @return array|mixed
     */
    function getTypes()
    {
        # скрываем выключенные, если не в FORDEV-режиме
        $sqlQuery = 'SELECT *, title_'.LNG.' AS title FROM '.TABLE_PUB_TYPES.' '.( !FORDEV ?' WHERE enabled = 1 ':'').' ORDER BY num';
        $c = $this->initTypeSettingsCache();
        $cacheKey = 'type-sett-all';

        if (!FORDEV) {
            if ( ( $aTypes = $c->get($cacheKey) ) === false ) {
                $aTypes = $this->db->select($sqlQuery);
                $aTypes = func::array_transparent($aTypes, 'keyword', true);
                foreach ($aTypes as $key=>$sett) {
                    $aTypes[$key] = new PublicationsSettings($sett);
                }
                $c->set($cacheKey, $aTypes);
            }
            return $aTypes;
        }

        $aTypes = $this->db->select($sqlQuery);
        $aTypes = func::array_transparent($aTypes, 'keyword', true);
        foreach ($aTypes as $key=>$sett) {
            $aTypes[$key] = new PublicationsSettings($sett);
        }
        $c->delete($cacheKey); # очищаем кеш в FORDEV-режиме
        return $aTypes;
    }

    /**
     * Получаем настройки типа публикации по ID
     * @param integer $nTypeID ID типа
     * @param boolean $bIgnoreCache игнорировать кеш
     * @param boolean $bReturnArray возвращать в виде массива
     * @return PublicationsSettings|array
     */
    function getTypeSettings($nTypeID, $bIgnoreCache = false, $bReturnArray = false)
    {
        static $cache = array();
        if (!$bIgnoreCache && isset($cache[$nTypeID])) {
            return $cache[$nTypeID];
        }

        $cKey = 'type-sett-'.$nTypeID;
        $c = $this->initTypeSettingsCache(); 
        if( $bIgnoreCache || ($aSettings = $c->get($cKey)) === false ) # ищем в кеше
        {
            $aSettings = $this->db->one_array('SELECT * FROM '.TABLE_PUB_TYPES.' WHERE id = :id', array(':id'=>$nTypeID));
            if (empty($aSettings)) return false;

            $this->db->langFieldsSelect($aSettings, $this->model->langTypes);
            PublicationsSettings::unpack($aSettings);
            if (!$bIgnoreCache) $c->set($cKey, $aSettings);
        }

        $cache[$nTypeID] = new PublicationsSettings($aSettings, false);
        return ( $bReturnArray ? $aSettings : $cache[$nTypeID] );
    }
    
    function initTypeSettingsCache()
    {
        return Cache::singleton($this->module_name, 'file');
    }
    
    function resetTypeSettingsCache($nTypeID = 0, $bAll = false)
    {
        # сбрасываем кеш настроек типа
        $cache = $this->initTypeSettingsCache();
        if($nTypeID > 0) {
            $cache->delete('type-sett-'.$nTypeID);
        }
        if($bAll) {
            $cache->delete('type-sett-all');
        }
    }

    public function setCurrentType($typeID)
    {
        $type = $this->getTypeSettings($typeID);
        if( ! empty($type))
        {
            $this->type = $type; # инициализируем настройки текущего типа
            $_GET['type'] = $_POST['type'] = $type->id;
            if ( ! bff::adminPanel()) {
                $this->typeTemplatesDir = $this->module_dir_tpl . DS . $type->keyword;
                $this->itemsTable = $this->getItemsTable($typeID);
            }
        }
    }

    function isKeywordExists($sKeyword, $sTable, $nExceptRecordID = null) //ss
    {
        return $this->db->one_data('SELECT id
                               FROM '.$sTable.'
                               WHERE '.( !empty($nExceptRecordID)? ' id!='.intval($nExceptRecordID).' AND ' : '' ).'
                                  keyword='.$this->db->str2sql($sKeyword).'  LIMIT 1');
    }
    
    function getKeyword($sKeyword='', $sTitle = '', $sTable, $nExceptRecordID = null) //ss
    {                                            
        if (empty($sKeyword) && !empty($sTitle)) {
            $sKeyword = mb_strtolower( func::translit( $sTitle ) );
        }
        $sKeyword = preg_replace('/[^a-zA-Z0-9]/','',$sKeyword);
        
        if (empty($sKeyword)) {
             $this->errors->set(_t('', 'Укажите keyword'));
        } else {
            if( $this->isKeywordExists($sKeyword, $sTable, $nExceptRecordID) )
             $this->errors->set(_t('','Указанный keyword уже используется'));
                                  
            if(!$nExceptRecordID) {
                if(is_dir(bff::path($sKeyword, 'images')) ||
                   $this->db->isTable(TABLE_PUB_ITEMS_.$sKeyword) ) 
                {
                    $this->errors->set(_t('','Указанный keyword уже используется'));
                }
            }
        }
        return $sKeyword;              
    }
    
    function deleteType($nTypeID)
    {
        if($nTypeID<=0 || !FORDEV) return false;
        
        $type = $this->getTypeSettings($nTypeID);
        if (empty($type)) return false;
        
        $key = $type->keyword;
        
        $this->manageTypeFiles($key, 'delete');
        
        # категории: свои или свои + общие(частично)
        if($type->categoriesCustomOnly() || $type->categoriesCustomAndCommon())
        {
            $this->db->exec('
                DELETE CL, C
                FROM '.TABLE_PUB_CATEGORIES.' C,
                     '.TABLE_PUB_CATEGORIES_LANG.' CL
                WHERE C.id = CL.id AND C.type_id = '.$nTypeID);
            $this->db->exec('DELETE FROM '.TABLE_PUB_CATEGORIES_IN.' WHERE type_id = '.$nTypeID);
        }
        
        # теги: свои
        if($type->tagsCustom()) {
            $this->db->exec('DELETE FROM '.TABLE_PUB_TAGS.' WHERE type_id = '.$nTypeID);
        }

        # чистим комментарии
        $this->db->exec('DELETE FROM '.TABLE_PUB_COMMENTS.' WHERE type_id = '.$nTypeID);

        # удаляем таблицу записей
        $this->db->exec('DROP TABLE IF EXISTS '. $this->getItemsTable($key) );

        $this->db->exec('DELETE FROM '.TABLE_PUB_ITEMS_LANG.' WHERE type_id = '.$nTypeID);

        $this->db->exec('DELETE FROM '.TABLE_PUB_TYPES.' WHERE id = '.$nTypeID);
    }
    
    function manageTypeFiles($sKeyword, $sAction)
    {
        switch ($sAction)
        {
            case 'create': 
            case 'delete':
            {
                if (empty($sKeyword)) {
                    $this->errors->set( _t('pub', 'Keyword типа указан некорректно') );
                    return false;
                }
                
                $aFolders = array(
                    'tpl' => $this->module_dir_tpl.DS.$sKeyword.DS,
                    'img' => bff::path($sKeyword, 'images'),
                );
                
                if($sAction == 'create') {
                    # создаем директории:
                    //$nChmod = 0666;
                    $nChmod = null;
                    mkdir($aFolders['tpl'], $nChmod); # для шаблонов
                    mkdir($aFolders['img'], $nChmod); # для изображений
                } else {
                    # удаляем директории:
                    rmdir( $aFolders['tpl'] ); # для шаблонов 
                    rmdir( $aFolders['img'] ); # для изображений     
                }
            } break;
            case 'rename': 
            {
                if(!empty($sKeyword['from']) && !empty($sKeyword['to']))
                {
                    # переименовываем директории:
                    rename( $this->module_dir_tpl.DS.$sKeyword['from'].DS,
                            $this->module_dir_tpl.DS.$sKeyword['to'].DS );
                    rename( bff::path($sKeyword['from'], 'images'),
                            bff::path($sKeyword['to'], 'images') );
                }
            } break;
        }
    }

    function manageTypeTables($sKeyword, $sAction)
    {
        if (empty($sKeyword)) {
            $this->errors->set( _t('pub', 'Keyword типа указан некорректно') );
            return false;
        }
        
        switch ($sAction)
        {
            case 'create':
            {
                $sqlInstall = Files::getFileContent($this->module_dir.'install.type.sql');
                $sqlInstall = strtr($sqlInstall, array('{key}'=>$sKeyword));
                $res = $this->db->exec($sqlInstall);
                return ($res===false ? false : true);
            } break;
            case 'rename': 
            {
                if(!empty($sKeyword['from']) && !empty($sKeyword['to']))
                {
                    if( $this->db->isTable(TABLE_PUB_ITEMS_.$sKeyword['from']) &&
                      ! $this->db->isTable(TABLE_PUB_ITEMS_.$sKeyword['to']) )
                    {
                        $res = $this->db->exec('RENAME TABLE '.TABLE_PUB_ITEMS_.$sKeyword['from'].' TO '.TABLE_PUB_ITEMS_.$sKeyword['to'].'');
                        return ($res===false ? false : true);
                    }
                    return false;
                }
            } break;
        }
    }
    
    # категории
    
    function categoriesProcessData($nTypeID = 0, $nCategoryID = 0)
    {
        $aParams = array(
            'pid'       => TYPE_UINT,
            'keyword'   => TYPE_NOTAGS, # keyword
            'enabled'   => TYPE_BOOL,   # включена ли категория
            'mtemplate' => TYPE_BOOL,   # использовать общий шаблон SEO
        );
        $aData = $this->input->postm($aParams);
        $this->input->postm_lang($this->model->langCategories, $aData);

        if (Request::isPOST())
        {
            $bSubCatsEnabled = true;
            if( $nTypeID > 0 ) {
                $oTypeSettings = $this->getTypeSettings($nTypeID);
                if( ! $oTypeSettings->categoriesCommonOnly() ) {
                    $aData['type_id'] = $nTypeID;
                }
                $bSubCatsEnabled = $oTypeSettings->subcategoriesEnabled();
            }
            if( ! $bSubCatsEnabled){
                $aData['pid'] = 0;
            }
            if($aData['title'] == '') {
                $this->errors->set( _t('pub', 'Название категории указано некорректно') );
            }
            if($aData['keyword'] == '') {
                $this->errors->set( _t('pub', 'Keyword категории указан некорректно') );
            } else {
                $nKeyWordExists = $this->db->one_data('
                    SELECT COUNT(*) FROM '.TABLE_PUB_CATEGORIES.'
                    WHERE keyword = :keyword AND pid = :pid AND id != :id ',
                    array(
                        ':id'       => $nCategoryID,
                        ':pid'      => $aData['pid'],
                        ':keyword'  => $aData['keyword'],
                    )
                );
                if($nKeyWordExists){
                    $this->errors->set( _t('pub', 'Keyword категории уже используется') );
                }
            }
        }
        return $aData;
    }
    
    function categoriesGetCurrentNum($nTypeID = 0, $bInheritItable = null, $nPid = 0)
    {
        if ($nPid) {
            $nNum = $this->db->one_data('
            SELECT MAX(num) FROM '.TABLE_PUB_CATEGORIES.'
            WHERE type_id = :type AND pid = :pid', array(
                    ':type' => $nTypeID,
                    ':pid'  => $nPid,
                ));
        } else {
            $nNum = $this->db->one_data('
                SELECT MAX(num) FROM '.($bInheritItable ? TABLE_PUB_CATEGORIES_IN : TABLE_PUB_CATEGORIES).'
                WHERE type_id = :type', array(
                    ':type' => $nTypeID
                ));
        }
        return intval($nNum) + 1;
    }

    /**
    * @param PublicationsSettings $oTypeSettings объект настроек типа публикации
    * @param boolean $bEnabled только влюченные категории
    * @param mixed $mOptions false - возвращать массив категорий, array(sel, empty) - возвращать html <options>
    * @param boolean $bCountItems подсчитывать ли кол-во записей в категориях
    * @return mixed (string | array)
    */
    function categoriesGet(PublicationsSettings $oTypeSettings = null, $bEnabled = true, $mOptions = false, $bCountItems = false)
    {
        $nPid = 0;
        if( ! empty($mOptions['pid'])){
            $nPid = $mOptions['pid'];
            unset($mOptions['pid']);
        }
        $bSub = false;
        if( ! empty($mOptions['sub'])){
            unset($mOptions['sub']);
            $bSub = true;
        }

        $bGetOptions = !empty($mOptions);
        $bCountItems = ($bCountItems && !empty($oTypeSettings) && !$bGetOptions);

        $sqlWhere = array();

        if (empty($oTypeSettings)) {
            $oTypeSettings = new PublicationsSettings( array('categories'=>3, 'subcategories' => 1), false ); //общие категории \/
        }
        if($bSub){
            $bSub = $oTypeSettings->subcategoriesEnabled();
        }

        switch ($oTypeSettings->categories)
        {
            case 0: # нет
            {
                return ( $bGetOptions ? '<option value="0">'._t('','нет доступных разделов').'</option>' : array());
            } break;
            case 1: # свои
            case 3: # только общие (полное наследование)
            {
                if($bGetOptions) $sqlSelect = array('C.id','C.pid','CL.title');
                else {
                    $sqlSelect = array('C.*','CL.*');
                    if($bCountItems) $sqlSelect[] = 'COUNT(I.id) as items';                    
                }

                $sqlSelect = join(',', $sqlSelect);
                                                            
                $sqlWhere[] = 'C.type_id = '.($oTypeSettings->categoriesCustomOnly() ? $oTypeSettings->id : 0);
                if($bEnabled) $sqlWhere[] = 'C.enabled = 1';
                if( ! $bSub){ $sqlWhere[] = 'C.pid = '.$nPid; }
                $aCategories = $this->db->select('SELECT '.$sqlSelect.', 0 as inherited
                        FROM '.TABLE_PUB_CATEGORIES.' C
                            '.($bCountItems ? ' LEFT JOIN '.$this->getItemsTable( $oTypeSettings->keyword ).' I ON I.category_id = C.id ' : '').'
                            , '.TABLE_PUB_CATEGORIES_LANG.' CL
                        WHERE '.join(' AND ', $sqlWhere).$this->db->langAnd(true, 'C', 'CL').'
                        '.($bCountItems ? ' GROUP BY C.id ':'').'
                        ORDER BY num');
            } break;
            case 2: # свои + общие (выборочное наследование)
            {
                if($bGetOptions) $sqlSelect = array('C.id','C.pid','CL.title');
                else {
                    $sqlSelect = array('C.*','CL.*','N.*');
                    if($bCountItems) $sqlSelect[] = 'COUNT(I.id) as items';
                }               
                
                $sqlSelect = join(',', $sqlSelect); 
                
                $sqlWhere[] = 'C.id = N.cat_id';
                $sqlWhere[] = 'N.type_id = '.$oTypeSettings->id;
                if($bEnabled) $sqlWhere[] = 'N.enabled = 1';
                if(! $bSub){ $sqlWhere[] = 'C.pid = '.$nPid; }
                $aCategories = $this->db->select('SELECT '.$sqlSelect.', (C.type_id != N.type_id) as inherited
                        FROM '.TABLE_PUB_CATEGORIES.' C, '.TABLE_PUB_CATEGORIES_LANG.' CL, '.TABLE_PUB_CATEGORIES_IN.' N
                            '.($bCountItems ? ' LEFT JOIN '.$this->getItemsTable( $oTypeSettings->keyword ).' I ON I.category_id = N.cat_id ' : '').'     
                        WHERE '.join(' AND ', $sqlWhere).$this->db->langAnd(true, 'C', 'CL').'
                        '.($bCountItems ? ' GROUP BY C.id ':'').'
                        ORDER BY N.num'); 
            } break;
        }
        if ($bSub && ! empty($aCategories)) {
            $aCategories = $this->db->transformRowsToTree($aCategories, 'id', 'pid', 'childnodes');
        }

        if($bGetOptions)
        {
            $sHTML = '';   
            if(!$mOptions['sel'] && $mOptions['empty']===false) $mOptions['empty'] = true;
            if(!empty($mOptions['empty'])) 
            {
                $sHTML .= '<option value="0">'.(is_string($mOptions['empty']) ? $mOptions['empty'] : _t('', 'Не указан')).'</option>';
            }
            foreach ($aCategories as $v) {
                if ($bSub && ! empty($v['childnodes'])){
                    $sHTML .= '<optgroup label="'.$v['title'].'">';
                    foreach ($v['childnodes'] as $vv){
                        $sHTML .= '<option style="padding-left:20px;" value="'.$vv['id'].'"'.($mOptions['sel']==$vv['id']?' selected="selected"':'').'>'.$vv['title'].'</option>';
                    }
                    $sHTML .= '</optgroup>';
                } else {
                    $sHTML .= '<option value="'.$v['id'].'"'.($mOptions['sel']==$v['id']?' selected="selected"':'').'>'.$v['title'].'</option>';
                }
            }
            return $sHTML;
        } else {    
            return ( ! bff::adminPanel() && ! $bSub ? func::array_transparent($aCategories, 'id', true) : $aCategories);
        }
    }

    /**
     * Инициализация компонента работы с тегами
     * @param integer $typeID ID типа
     * @return PublicationsTags
     */
    public function initTags($typeID)
    {
        static $i;
        if (!isset($i)) {
            $i = new PublicationsTags();
        }
        $i->setType($typeID);

        return $i;
    }

    # календарь
    /**
     * Фиксируем наличие публикаций за указанную дату
     * @param integer $nTypeID ID типа публикации
     * @param mixed $mDate (string = '2006-04-05 01:50:00', integer) дата
     * @param integer $nCityID ID города
     * @param integer $nIncrement наличие: +1, -1
     * @return boolean
     */
    function calendarUpdate($nTypeID, $mDate, $nCityID, $nIncrement = 1)
    {
        # Исключаем некоторые типы публикаций из календаря:
        if (in_array($nTypeID, array(static::TYPE_ARTICLES, static::TYPE_VIDEO))) {
            return true;
        }
        if (is_string($mDate)) {
            $mDate = func::parse_datetime($mDate);
            $mDate = intval("{$mDate['year']}{$mDate['month']}{$mDate['day']}");
        }
        $res = $this->db->exec('INSERT INTO '.TABLE_PUB_CALENDAR.' (calendar_date, items, city_id)
            VALUES ('.$mDate.', 1, 0)
            ON DUPLICATE KEY UPDATE items = items + '.(int)$nIncrement.'
        ');
        
        return !empty($res);
    }
    
    /**
     *
     */
    function calendarGetActiveDates($mSelectedDate = false, $bJSON = true)
    {
        $n31Days = 2678400; //60*60*24*31
        $nDate = ($mSelectedDate !== false ? (is_string($mSelectedDate) ? strtotime($mSelectedDate) : $mSelectedDate) : time());
        $nDate = mktime(0,0,0,date('m',$nDate),1,date('y',$nDate));
        if($mSelectedDate!==false) {
            //возвращаем данные за: требуемый месяц
            $nFrom = (int)date('Ymd', $nDate);
            $nTo   = (int)date('Ymd', $nDate + $n31Days);
        } else {
            //возвращаем данные за: предыдущий / текущий / следующий месяца
            $nFrom = (int)date('Ymd', $nDate - $n31Days);
            $nTo   = (int)date('Ymd', $nDate + ($n31Days * 2));
        }

        $res = $this->db->select_one_column('SELECT calendar_date FROM '.TABLE_PUB_CALENDAR.' 
            WHERE calendar_date>= '.$nFrom.' AND calendar_date<='.$nTo.' AND items>0');
        if (empty($res)) $res = array();

        return ($bJSON ? '['.join(',',$res).']' : $res);
    }
    
    # RSS для Yandex
    protected function rssBuild_Yandex($nLimit = 10)
    {
        # Общее описание
        $sDescription = '
            <title>'._t('pub', 'Городской портал').' '.SITEHOST.'</title>
            <link>'.static::url('list', self::TYPE_NEWS).'</link>
            <language>'.$this->locale->getLanguageSettings(LNG, 'locale').'</language>
            <description>'._t('','Информационный портал области').'</description>
            <yandex:logo type="square">'.SITEURL_STATIC.'/img/logo.gif</yandex:logo>
            <image>
                <url>'.SITEURL_STATIC.'/img/logo.gif</url>
                <title>'._t('','Городской портал').' '.SITEHOST.'</title>
                <link>'.bff::urlBase().'</link>
            </image>            
        ';
        
        $sql = array();   
        $sqlLimit = $this->db->prepareLimit(0, $nLimit);
        foreach( array(self::TYPE_NEWS, self::TYPE_REPORTS, self::TYPE_ARTICLES, self::TYPE_NAROD, ) as $type) {
            $key = $this->typeKey($type);
            $sql[] = '(SELECT I.id, IL.title, I.link, IL.content_search, IL.content_short as description, I.img, I.publicated, IL.author,
                            CL.title as cat_title,
                            '.$this->db->str2sql($key).' as type_key,
                            '.$this->db->str2sql($type).' as type_id
                        FROM '.TABLE_PUB_ITEMS_.$key.' I,
                             '.TABLE_PUB_ITEMS_LANG.' IL,
                             '.TABLE_PUB_CATEGORIES_LANG.' CL
                        WHERE I.enabled = 1 AND I.extra_xml_yandex = 1 AND CL.id = I.category_id AND CL.lang = '.$this->db->str2sql(LNG).'
                        '.($type == self::TYPE_NAROD ? ' AND I.moderated = 1 ' : '').
                        $this->model->langAnd($type, true, 'I', 'IL').
                        ' ORDER BY I.publicated DESC '
                        .$sqlLimit.')';
        }

        # Публикации
        $aItems = $this->db->select( join(' UNION ', $sql) . ' ORDER BY publicated DESC ' . $sqlLimit);

        if(!empty($aItems))
        {
            $aTypes = $this->getTypes(); 
            
            foreach($aItems as $k=>$v)
            {
                if( ! empty($v['img'])) {
                    $imgType = Files::getExtension($v['img']);
                    $imgType = 'image/' . ( $imgType == 'jpg' ? 'jpeg' : $imgType );
                    $imgSizePrefix = PublicationsImages::szNormal;
                    switch ($v['type_id'])
                    {
                        case static::TYPE_NAROD: {
                            $imgSizePrefix = false;
                        } break;
                    }
                    $v['img'] = $this->getImagesPath(true, array('id'=>$v['id'], 'publicated'=>$v['publicated'], 'filename'=>$v['img'], 'size'=>$imgSizePrefix), $aTypes[$v['type_key']]);
                }

                //pubDate = Fri, 18 Nov 2011 17:37:20 +0300
                $aItems[$k] = '<item>
                               <title>'.htmlspecialchars(strip_tags($v['title'])).'</title>
                               <link>'.htmlspecialchars('http:'.static::url('view', $v['link'])).'</link>
                               <yandex:full-text>'.htmlspecialchars($v['description'].' '.$v['content_search']).'</yandex:full-text>
                               <category>'.$v['cat_title'].'</category>
                               '.(!empty($v['img']) ? '<enclosure url="'.$v['img'].'" type="'.$imgType.'"/>' : '').'
                               <pubDate>'.date('D, j M Y G:i:s O', strtotime($v['publicated'])).'</pubDate>
                               '.(!empty($v['author']) ? '<author>'.htmlspecialchars($v['author']).'</author>' : '').'
                               </item>';
            }
        } else {
            $aItems = array();
        }

        return strtr('<?xml version="1.0" encoding="UTF-8"?><rss xmlns:yandex="http://news.yandex.ru" xmlns:media="http://search.yahoo.com/mrss/" version="2.0"><channel>{description}{items}</channel></rss>',
            array(
                '{description}'=>$sDescription, 
                '{items}'=>join('', $aItems)
                )
            );
    }

    # RSS для раздела "News"
    protected function rssBuild_Type($nTypeID, $nLimit = 10, $aCategoryData = false)
    {
        $bCategory = !empty($aCategoryData);
        $type = $this->getTypeSettings($nTypeID);
        if ($bCategory) {
            $link = static::url('category', array('type'=>$nTypeID, 'keyword'=>$aCategoryData['keyword']));
        } else {
            $link = static::url('list', $nTypeID);
        }
        
        # Общее описание
        $sDescription = '
            <title>'.($bCategory ? $aCategoryData['title'].' - ':'')._t('', 'Городской портал').' '.SITEHOST.'</title>
            <link>'.$link.'</link>
            <language>'.$this->locale->getLanguageSettings(LNG, 'locale').'</language>
            <description>'._t('','Информационный портал области').'</description>
            <image> 
                <url>'.SITEURL_STATIC.'/img/logo.gif</url>
                <title>'._t('','Городской портал').' '.SITEHOST.'</title>
                <link>'.bff::urlBase().'</link>
            </image>            
        ';
        
        # Публикации
        $aItems = $this->db->select('SELECT I.id, IL.title, I.link, IL.content_short as description,
                                            I.img, I.publicated, IL.author, CL.title as cat_title
                                     FROM '.$this->getItemsTable($nTypeID).' I,
                                          '.TABLE_PUB_ITEMS_LANG.' IL,
                                          '.TABLE_PUB_CATEGORIES_LANG.' CL
                                     WHERE I.enabled = 1
                     '.($bCategory ? ' AND I.category_id = '.$aCategoryData['id'] : '').'
                                       AND CL.id = I.category_id AND CL.lang = :lng'.
                                       $this->model->langAnd($nTypeID, true, 'I', 'IL').'
                                     ORDER BY I.publicated DESC
                                     '.$this->db->prepareLimit(0, $nLimit), array(':lng'=>LNG));

        if( ! empty($aItems))
        {
            foreach($aItems as $k=>$v)
            {
                if(!empty($v['img'])) {
                    $imgType = Files::getExtension($v['img']);
                    $imgType = 'image/' . ( $imgType == 'jpg' ? 'jpeg' : $imgType );
                    $v['img'] = $this->getImagesPath(true, array('id'=>$v['id'], 'publicated'=>$v['publicated'], 'filename'=>$v['img'], 'size'=>PublicationsImages::szNormal), $type);
                }
                
                //pubDate = Fri, 18 Nov 2011 17:37:20 +0300
                $aItems[$k] = '<item>
                               <title>'.htmlspecialchars(strip_tags($v['title'])).'</title>
                               <link>'.htmlspecialchars('http:'.static::url('view', $v['link'])).'</link>
                               <description>'.htmlspecialchars($v['description']).'</description>
                               <category>'.$v['cat_title'].'</category>
                               '.(!empty($v['img']) ? '<enclosure url="'.$v['img'].'" type="'.$imgType.'"/>' : '').'
                               <pubDate>'.date('D, j M Y G:i:s O', strtotime($v['publicated'])).'</pubDate>
                               '.(!empty($v['author']) ? '<author>'.htmlspecialchars($v['author']).'</author>' : '').'
                               </item>';
            }
        } else {
            $aItems = array();
        }

        return strtr('<?xml version="1.0" encoding="UTF-8"?><rss version="2.0"><channel>{description}{items}</channel></rss>',
            array(
                '{description}'=>$sDescription, 
                '{items}'=>join('', $aItems)
                )
            );
    }
    
    function rssUpdate($nTypeID, $nCategoryID = 0)
    {
        // Строим общий файл rss (для яндекса)
        $sXML = $this->rssBuild_Yandex(20);
        Files::putFileContent(PATH_PUBLIC.'rss/yandex-news.xml', $sXML);
        
        # если изменялись основные новости "News"
        # перестраиваем /rss/news
        if($nTypeID == self::TYPE_NEWS)
        {
            # перестраиваем rss категории
            if($nCategoryID > 0) 
            {
                $aCategoryData = $this->db->one_array('
                    SELECT C.id, CL.title, C.keyword
                    FROM '.TABLE_PUB_CATEGORIES.' C,
                         '.TABLE_PUB_CATEGORIES_LANG.' CL
                    WHERE C.id = '.$nCategoryID.$this->db->langAnd(true, 'C', 'CL'));
                if(!empty($aCategoryData)) {
                    $xmlCategory = $this->rssBuild_Type(self::TYPE_NEWS, 20, $aCategoryData);
                    Files::putFileContent(PATH_PUBLIC.'rss/news/'.$aCategoryData['keyword'].'.xml', $xmlCategory);
                }
            }
            
            # перестариваем общий RSS
            $xmlNews = $this->rssBuild_Type(self::TYPE_NEWS, 20, false);
            Files::putFileContent(PATH_PUBLIC.'rss/news/all.xml', $xmlNews);
        }
        
        return true;
    }

    /**
     * Обработка ситуации c необходимостью ре-формирования URL
     */
    public function onLinksRebuild()
    {
        $this->model->itemsLinksRebuild();
    }

    /**
     * Формирование списка директорий/файлов требующих проверки на наличие прав записи
     * @return array
     */
    public function writableCheck()
    {
        foreach($this->getTypes() as $k=>$v) {
            $res[bff::path($k, 'images')] = 'dir'; # файлы изображений типа
        }

        $res[PATH_PUBLIC.'rss'] = 'dir'; # rss файлы
        $res[PATH_PUBLIC.'rss'.DS.'yandex-news.xml']   = 'file-e'; # yandex-news.xml
        $res[PATH_PUBLIC.'rss'.DS.'news'.DS.'all.xml'] = 'file-e'; # news-all.xml
        $aCats = $this->db->select('
            SELECT C.keyword
            FROM '.TABLE_PUB_CATEGORIES.' C
            WHERE C.type_id = 0 AND C.pid = 0 AND C.enabled = 1
            ORDER BY C.num');

        foreach($aCats as $v) {
            $v = PATH_PUBLIC.'rss'.DS.'news'.DS.$v['keyword'].'.xml';
            if (file_exists($v)) {
                $res[$v] = 'file-e';
            }
        }

        if (FORDEV) {
            # доступность создания директорий для новых типов публикаций
            $res[$this->module_dir] = 'dir';
            $res[$this->module_dir_tpl] = 'dir';
        }

        return array_merge(parent::writableCheck(), $res);
    }
}