<?php

abstract class GeoBase extends GeoModule
{
    /** @var GeoModel */
    public $model = null;
    /** @var array доступные для редактирования (задействованные) geo-сущности */
    public static $useRegions = array(
        self::lvlCountry,
        self::lvlCity,
        self::lvlDistrict,
    );

    /**
     * Тип формирования URL с учетом региона/города
     */
    const URL_SUBDOMAIN = 1; # поддомены - city.example.com
    const URL_NONE      = 3; # город не фигурирует в URL - example.com

    /**
     * Тип покрытия регионов
     */
    const COVERING_CITIES = 3; # несколько городов

    /**
     * Инициализируем фильтр по региону
     * @param mixed $key ключ требуемых данных о текущих настройках фильтра по региону или FALSE (все данные)
     * @return mixed
     */
    public static function filter($key = false)
    {
        static $inited = false, $current = array(), $city = array();
        if (!$inited) {
            $inited = true;

            # для некоторых страниц регион в URL не должен влиять на выбранный пользователем
            $url = static::filterUrl();
            $user = 0;
            $userCover = true;
            $userRegionChanging = (isset($_GET['region']) || isset($_POST['region']));
            $userRegionID = bff::input()->getpost('region', TYPE_UINT);

            if (bff::isIndex() || static::cityIsOne()) {
                if ($url['id']) {
                    static::filterUser($url['id']);
                    $city = $url['city'];
                    $userCover = false;
                } else {
                    if ($userRegionChanging) {
                        static::filterUser($userRegionID);
                    }
                    $user = static::regionData(static::filterUser());
                }
            } else {
                if (!$userRegionChanging) {
                    if ($url['id'] && bff::$isBot) {
                        $city = $url['city'];
                    }
                    $user = static::regionData(static::filterUser());
                } else {
                    $city = $url['city'];

                    static::filterUser($userRegionID);
                    if ($userRegionID) {
                        $user = static::regionData($userRegionID);
                    }
                }
            }

            if ($userCover && $user && !bff::$isBot) {
                if (static::coveringRegionCorrect($user)) {
                    $city = $user;
                }
            }

            $current = ($city ? $city : array('id' => 0));
        }

        switch ($key) {
            case 'id':
            { # id текущего: города | 0
                return (!empty($current['id']) ? $current['id'] : 0);
            }
            break;
            case 'url':
            { # данные о городе
                return array(
                    'city'   => ($city ? $city['keyword'] : ''),
                );
            }
            case 'title':
            {
                return (!empty($current['title_'.LNG]) ? $current['title_'.LNG] : '');
            } break;
            break;
            default:
                return $current;
        }
    }

    /**
     * Получаем ID города на основе текущего URL
     * @param mixed $key ключ требуемых данных
     * @return integer|array
     */
    public static function filterUrl($key = false)
    {
        static $inited = false, $city = 0, $id = 0, $keyword = '';
        if (!$inited) {
            $inited = true;
            switch (static::urlType()) {
                case self::URL_SUBDOMAIN: # поддомены
                {
                    $host = Request::host(SITEHOST);
                    if (preg_match('/(.*)\.' . preg_quote(SITEHOST) . '/', $host, $matches) > 0 && !empty($matches[1])) {
                        # город
                        if (strpos($matches[1], '.')) { # {city}.{module}
                            list(,$matches[1]) = explode('.', $matches[1], 2);
                        }
                        $data = static::regionDataByKeyword($matches[1]);
                        if ($data && !static::coveringRegionCorrect($data)) {
                            $data = false;
                        }
                        if ($data) {
                            if ($data['numlevel'] == self::lvlCity) {
                                $city = $data;
                            }
                        }
                    }
                }
                break;
                case self::URL_NONE: # регион не фигурирует в URL
                {
                    # не анализируем URL
                }
                break;
            }

            list($id, $keyword) = ($city ? array($city['id'], $city['keyword']) : array(0, ''));
        }

        if ($key!==false) {
            switch ($key)
            {
                case 'id':      return $id;
                case 'keyword': return $keyword;
                case 'title':   return ( ! empty($city['title_'.LNG]) ? $city['title_'.LNG] : '' );
            }
        }

        return array('city' => $city, 'id' => $id);
    }

    /**
     * Получаем / устанавливаем ID города пользователя
     * @param integer|boolean $nCityID устанавливаем ID города или FALSE (получаем текущий)
     * @return integer
     */
    public static function filterUser($nCityID = false)
    {
        if (bff::$isBot) {
            if ($nCityID !== false) {
                return $nCityID;
            } else {
                return 0;
            }
        }

        $cookieKey = config::sys('cookie.prefix') . 'geo';
        $cookieExpire = 100; # кол-во дней

        if ($nCityID !== false) {
            # обновляем куки
            if (!isset($_COOKIE[$cookieKey]) || $_COOKIE[$cookieKey] != $nCityID) {
                Request::setCOOKIE($cookieKey, $nCityID, $cookieExpire);
                $_COOKIE[$cookieKey] = $nCityID;
            }
        } else {
            if (!isset($_COOKIE[$cookieKey])) {
                # определяем ID города по IP адресу пользователя
                if (config::sys('geo.ip.location', false)) {
                    $aData = static::regionData(3); //static::model()->regionDataByIp();
                    if (empty($aData)) {
                        $aData = array('id' => 0);
                    } else {
                        if (!static::coveringRegionCorrect($aData)) {
                            $aData['id'] = 0;
                        }
                    }

                    $nCityID = $aData['id'];
                    if ($nCityID > 0) {
                        # кешируем
                        static::regionData($aData['id'], $aData);
                        static::regionDataByKeyword($aData['keyword'], $aData);
                    }
                } else {
                    $nCityID = 0;
                }
                $_COOKIE[$cookieKey] = $nCityID;
                Request::setCOOKIE($cookieKey, $nCityID, $cookieExpire);
            } else {
                # получаем из куков
                $nCityID = bff::input()->cookie($cookieKey, TYPE_UINT);
            }
        }

        return $nCityID;
    }

    /**
     * Текущий тип формирования URL с учетом региона и города
     * @return mixed
     */
    public static function urlType()
    {
        return config::get('geo_url', self::URL_NONE);
    }

    public static function url($cityID = false)
    {
        if ($cityID === false) {
            $cityID = static::cityID();
        }
        return static::urlBase(LNG, false, array('city'=>$cityID));
    }

    /**
     * Текущий тип покрытия регионов
     * @param int|array $type >0|array - проверяем на соответствие, 0 - возвращаем текущий
     * @return bool|mixed
     */
    public static function coveringType($type = 0)
    {
        $current = (int)config::get('geo_covering', self::COVERING_CITIES);
        if (!empty($type)) {
            if (is_array($type)) {
                return in_array($current, $type);
            }

            return ($current == $type);
        }

        return $current;
    }

    /**
     * ID города/нескольких городов в зависимости от текущего покрытия регионов
     * @param int $regionLevel >0 - на основе Geo::lvl_, 0 - в зависимости от текущего покрытия регионов
     * @return bool|mixed
     */
    public static function coveringRegion($regionLevel = 0)
    {
        if (!$regionLevel) {
            switch (static::coveringType()) {
                case self::COVERING_CITIES:
                default:
                {
                    $data = strval(config::get('geo_covering_lvl' . self::lvlCity, ''));
                    if (strpos($data, ',') !== false) {
                        $data = explode(',', $data);
                        $data = array_map('intval', $data);
                    } else {
                        $data = array(intval($data));
                    }

                    return $data;
                }
                break;
            }
        } else {
            return config::get('geo_covering_lvl' . $regionLevel,
                ($regionLevel == self::lvlCountry ? static::defaultCountry() : 0)
            );
        }
    }

    /**
     * Проверка данных текущего региона на соответствие текущим настройка типа покрытия регионов
     * @param array $data @ref данные о текущем региона
     * @return boolean
     */
    public static function coveringRegionCorrect(array &$data)
    {
        if (empty($data) || !$data['id']) {
            return true;
        }

        switch (static::coveringType()) {
            case self::COVERING_CITIES:
            {
                if (!in_array($data['id'], static::coveringRegion())) {
                    return false;
                }
            }
            break;
        }

        return true;
    }

    /**
     * Формируем список доступных стран
     * @param boolean $resetCache сбросить кеш
     * @return array
     */
    public static function countryList($resetCache = false)
    {
        $cacheKey = 'country-list';
        $cache = Cache::singleton('geo');
        if ($resetCache === true || ($aData = $cache->get($cacheKey)) === false)
        {
            $aData = static::model()->regionsList(static::lvlCountry, array('enabled'=>1));
            $cache->set($cacheKey, $aData);
        }
        return $aData;
    }

    /**
     * Формируем список доступных стран в формате select::options
     * @param int|bool $mSelectedID
     * @param mixed $mEmptyOption название option-пункта в случае если страна не указана
     * @return string
     */
    public static function countryOptions($mSelectedID = false, $mEmptyOption = 'Выбрать')
    {
        $aData = self::countryList();
        if (empty($mSelectedID)) {
            $mSelectedID = static::defaultCountry();
        }
        return HTML::selectOptions($aData, $mSelectedID, $mEmptyOption, 'id', 'title');
    }

    /**
     * Получаем данные о текущей стране
     * @param string $sKey ключ требуемых данных
     * @param integer $nCountryID ID страны или 0 - страна по-умолчанию
     * @return string
     */
    public static function countryData($sKey = '', $nCountryID = 0)
    {
        # Получаем страну по-умолчанию
        if ( ! $nCountryID || $nCountryID <=0 ) {
            $nCountryID = ( static::defaultCountry() ? static::defaultCountry() : self::cityData('country') );
        }
        $aData = static::regionData($nCountryID);
        return ( ! empty($sKey) && isset( $aData[$sKey] ) ? $aData[$sKey] : $aData );
    }

    /**
     * Формируем список городов
     * @param bool $noCity
     * @return array
     */
    public static function cityList($noCity = false)
    {
        static $data;
        if (!isset($data)) {
            $cache = Cache::singleton('geo');
            $cacheKey = 'city-list-'.LNG;
            if (($data = $cache->get($cacheKey)) === false)
            {
                $filter = array();
                if (static::coveringType(static::COVERING_CITIES)) {
                    $filter['id'] = static::coveringRegion();
                } else {
                    $filter['enabled'] = 1;
                    $filter['country'] = static::defaultCountry();
                }
                $data = static::model()->regionsList(static::lvlCity, $filter);
                $cache->set($cacheKey, $data);
            }
        }
        if ($noCity && ! static::cityIsOne()) {
            $data = array_reverse($data, true);
            $data[0] = array('id'=>0, 'title'=>_t('geo', 'Все города'), 'keyword'=>'');
            $data = array_reverse($data, true);
        } else {
            if (isset($data[0])) {
                unset($data[0]);
            }
        }
        return $data;
    }

    /**
     * Проверяем, задействован ли только 1 город, или несколько
     * @return bool
     */
    public static function cityIsOne()
    {
        return (sizeof(static::cityList()) < 2);
    }

    /**
     * Проверяем на корректность указанный пользователем город
     * @param integer $nCityID ID города
     * @return bool
     */
    public static function cityIsValid($nCityID)
    {
        if (empty($nCityID) || $nCityID <=0) return false;
        $aCityList = static::cityList();
        if (!empty($aCityList)) {
            return array_key_exists($nCityID, $aCityList);
        }
        return static::isCity($nCityID);
    }

    /**
     * Формируем список доступных городов в формате select::options
     * @param int|bool $mSelectedID
     * @param mixed $mEmptyOption название option-пункта в случае если город не указан
     * @return string
     */
    public static function cityOptions($mSelectedID = false, $mEmptyOption = 'Выбрать')
    {
        $aData = static::cityList();
        if ($mSelectedID === false || ($mSelectedID === 0 && static::regionsFilter() && ! bff::adminPanel())) {
            $mSelectedID = static::cityID();
        }
        return HTML::selectOptions($aData, $mSelectedID, $mEmptyOption, 'id', 'title');
    }

    /**
     * Формируем поле выбора города, с учетом кол-ва городов и его обязательности
     * @param integer|bool $mSelectedID ID выбранного города; false или 0 - подставляем текущий город пользователя
     * @param bool $bRequired обязателен ли город
     * @param string $sFieldName имя поля
     * @param array $aAttributes дополнительные атрибуты для поля select
     * @param mixed $mEmptyOption название option-пункта в случае если город не указан
     * @return string HTML <select>...</select>
     */
    public static function cityOptionsSelect($mSelectedID, $bRequired, $sFieldName, array $aAttributes = array(), $mEmptyOption = 'Не важно')
    {
        $aCityList = static::cityList();
        if (!empty($sFieldName)) $aAttributes['name'] = $sFieldName;
        $aDataAttributes = array('ycoords', 'metro');
        if (sizeof($aCityList) < 2) {
            # город всего один
            $aAttributes['style'] = 'display:none;';
            $sResult  = HTML::selectOptions($aCityList, static::defaultCity(), false, 'id', 'title', $aDataAttributes);
            $sResult  = '<select'.HTML::attributes($aAttributes).'>'.$sResult.'</select>';
            $sResult .= '<strong>'.static::regionTitle(static::defaultCity()).'</strong>&nbsp;';
        } else {
            # городов несколько
            if ($bRequired && ! $mSelectedID && ! bff::adminPanel()) {
                $mSelectedID = static::cityID();
            }
            $sResult = HTML::selectOptions($aCityList, $mSelectedID, ( $bRequired ? false : $mEmptyOption), 'id', 'title', $aDataAttributes);
            $sResult = '<select'.HTML::attributes($aAttributes).'>'.$sResult.'</select>';
        }
        return $sResult;
    }

    /**
     * Формируем список доступных городов (в указанной стране) в формате select::options
     * @param int $nCountryID ID страны
     * @param int|bool $mSelectedID
     * @param mixed $mEmptyOption название option-пункта в случае если город не указан
     * @return string
     */
    public static function cityOptionsByCountry($nCountryID, $mSelectedID = false, $mEmptyOption = 'Выбрать')
    {
        $aFilter = array(
            'enabled' => 1,
            'country' => ( ! empty($nCountryID) && $nCountryID > 0 ? $nCountryID : static::defaultCountry() ),
        );
        $aData = static::model()->regionsList(static::lvlCity, $aFilter);
        if (empty($mSelectedID)) {
            $mSelectedID = self::cityID();
        }
        return HTML::selectOptions($aData, $mSelectedID, $mEmptyOption, 'id', 'title');
    }

    /**
     * Получаем ID текущего города
     * @param bool $default если текущий город не указан - брать город по-умолчанию
     * @return integer
     */
    public static function cityID($default = false)
    {
        if ( ! $default) {
            return static::filter('id');
        } else {
            $id = static::filter('id');
            if ( ! $id) {
                $id = static::defaultCity();
            }
            return $id;
        }
    }

    /**
     * Получаем данные о текущем городе
     * @param string $sKey ключ требуемых данных
     * @param integer $nCityID ID города или 0 - город по-умолчанию
     * @return string
     */
    public static function cityData($sKey = '', $nCityID = 0)
    {
        # Получаем город по-умолчанию
        if ($nCityID<=0) {
            $nCityID = static::defaultCity();
        }

        $aData = static::regionData($nCityID);
        return ( ! empty($sKey) && isset($aData[$sKey]) ? $aData[$sKey] : $aData );
    }

    /**
     * ID города по-умолчанию
     * @return mixed
     */
    public static function defaultCity()
    {
        $id = key(static::cityList());
        if ( ! $id) $id = parent::defaultCity();
        return $id;
    }

    /**
     * Формируем список районов города
     * @param int $nCityID ID города
     * @param bool $resetCache сбросить кеш
     * @return array
     */
    public static function districtList($nCityID = 0, $resetCache = false)
    {
        $cacheKey = 'district-list-'.$nCityID;
        $cache = Cache::singleton('geo');
        if ($resetCache === true || ($aData = $cache->get($cacheKey)) === false)
        {
            $aData = static::model()->districtsList($nCityID);
            $cache->set($cacheKey, $aData);
        }
        return $aData;
    }

    /**
     * Формируем список районов города в формате select::options
     * @param int $nCityID ID города
     * @param int|bool $mSelectedID
     * @param mixed $mEmptyOption название option-пункта в случае если район не указан
     * @return string
     */
    public static function districtOptions($nCityID = 0, $mSelectedID = false, $mEmptyOption = 'Выбрать')
    {
        return HTML::selectOptions(self::districtList($nCityID), $mSelectedID, $mEmptyOption, 'id', 't');
    }

    /**
     * Сбрасываем кеш
     * @param mixed $mLevel тип региона Geo::lvl...
     * @param string $mExtra
     */
    public function resetCache($mLevel = false, $mExtra = '')
    {
        Cache::singleton('geo')->flush('geo');
    }

    /**
     * Формируем подсказку(presuggest) состоящую из основных городов, в формате JSON
     * @param int $nCountryID ID страны или 0 - Geo::defaultCountry()
     * @param bool $bReturnArray вернуть результат в виде массива
     */
    public static function regionPreSuggest($nCountryID = 0, $bReturnArray = false)
    {
        if (empty($nCountryID)) $nCountryID = static::defaultCountry();

        $cache = Cache::singleton('geo');
        $cacheKey = 'city-presuggest-'.$nCountryID.'-'.LNG;
        if ( ($aData = $cache->get($cacheKey)) === false )
        {
            # получаем список предвыбранных основных городов страны
            $aData = static::model()->regionsList(self::lvlCity, array('main>0','enabled'=>1,'country'=>$nCountryID), 0, 15, 'main');
            $cache->set($cacheKey, $aData);
        }
        if ($bReturnArray) {
            return $aData;
        }

        $aResult = array();
        foreach ($aData as $v) {
            $aResult[] = array($v['id'], $v['title'], $v['metro']);
        }

        return func::php2js($aResult); # возвращаем в JSON-формате для autocomplete.js
    }

    /**
     * Координаты карты по-умолчанию
     * @param boolean $explode вернуть в качестве массива
     * @return string|array
     */
    public static function mapsDefaultCoords($explode = false)
    {
        $coords = static::cityData('ycoords', static::cityID(true));
        if (empty($coords) || strpos($coords, ',') === false) {
            $coords = config::sys('geo.maps.default.coords', static::$ymapsDefaultCoords);
        }
        return ($explode ? explode(',', $coords) : $coords);
    }

    /**
     * Проверка координат, если не указаны, корректируем на координаты по-умолчанию
     * @param mixed $lat @ref координата latitude
     * @param mixed $lon @ref координата longitude
     */
    public static function mapsDefaultCoordsCorrect(&$lat, &$lon)
    {
        if (!floatval($lat) || !floatval($lon)) {
            list($lat, $lon) = static::mapsDefaultCoords(true);
        }
    }

   /**
    * Тип карт
    * @param boolean $includeEditor подключать редактор карты
    * @return mixed
    */
    public static function mapsAPI($includeEditor = false)
    {
        switch (static::mapsType())
        {
            case self::MAPS_TYPE_YANDEX:
                tpl::includeJS(static::$ymapsJS, false);
                break;
            case self::MAPS_TYPE_GOOGLE:
                tpl::includeJS(Request::scheme().'://maps.googleapis.com/maps/api/js?key='.config::sys('geo.maps.googleKey','').'&v=3&sensor=false&language='.LNG, false);
                break;
        }
        if ($includeEditor) {
            tpl::includeJS('maps.editor', true);
        }
    }

    /**
     * Включена ли фильтрация по региону в модулях
     */
    public static function regionsFilter()
    {
        return (
            # фильтрация выключена в системных настройках
            (bool)config::sys('geo.regions.filter', false) &&
            # задействовано более одного города
            ! static::cityIsOne()
        );
    }

}