<?php

abstract class BillsBase extends BillsModule
{
    # доп. системы оплаты:
    const PS_PRIVAT24 = 32; # Privat24

    const PS_BALANCE  = 'balance'; # Внутренний счет

    /** @var BillsModel */
    public $model = null;

    function init()
    {
        parent::init();

        $this->module_title = 'Счета';

        # включаем доступные системы оплаты:
        $this->psystemsAllowed = array(
            self::PS_ROBOX,
            self::PS_WM,
            self::PS_W1,
            self::PS_PRIVAT24,
        );

        $this->psystemsData[self::PS_PRIVAT24] = array(
            'id'=>self::PS_PRIVAT24, 'title'=>'Privat24', 'key'=>'privat24', 'desc'=>''
        );

        /**
         * Настройки доступных систем оплаты указываются в файле /config/sys.php
         * Полный список доступных настроек указан в BillsModuleBase::init методе
         * Формат: 'bills.[ключ системы оплаты].[ключ настройки]'
         * Пример: 'bills.robox.test' - тестовый режим системы оплаты Robokassa
         *
         * URL для систем оплат:
         * Result:  http://example.com/bill/process/(robox|wm)
         * Success: http://example.com/bill/success
         * Fail:    http://example.com/bill/fail
         */
    }

    /**
     * Формирование URL
     * @param string $key ключ
     * @param mixed $opts доп. параметры
     * @param boolean $dynamic динамическая ссылка
     * @return string
     */
    public static function url($key, $opts = array(), $dynamic = false)
    {
        $base = static::urlBase(LNG, $dynamic);
        switch ($key) {
            # История операций
            case 'history':
                return $base . '/bill/history' . static::urlQuery($opts);
                break;
            # Пополнение счета
            case 'balance':
                return $base . '/bill/balance' . static::urlQuery($opts);
                break;
            # Process
            case 'process':
                return $base . '/bill/process' . (isset($opts['ps']) ? $opts['ps'] : '') . static::urlQuery($opts, array('ps'));
                break;
            # Success
            case 'success':
                return $base . '/bill/success' . static::urlQuery($opts);
                break;
            # Fail
            case 'fail':
                return $base . '/bill/fail' . static::urlQuery($opts);
                break;
        }
    }

    public static function getPaySystems($bBalanceUse = false)
    {
        $aData = array(
            'robox'    => array(
                'id'           => self::PS_ROBOX,
                'class'        => 'ps-robox',
                'way'          => '',
                'title'        => _t('bills', 'Robokassa'),
                'currency_id'  => 1, # рубли
                'enabled'      => true,
            ),
            'wm'       => array(
                'id'           => self::PS_WM,
                'class'        => 'ps-wm',
                'way'          => 'wmz',
                'title'        => _t('bills', 'Webmoney'),
                'currency_id'  => 3, # доллары
                'enabled'      => true,
            ),
            'w1' => array(
                'id'           => self::PS_W1,
                'class'        => 'ps-w1',
                'way'          => '',
                'title'        => _t('bills', 'W1'),
                'currency_id'  => 1, # рубли
                'enabled'      => true,
            ),
            'privat24' => array(
                'id'           => self::PS_PRIVAT24,
                'class'        => 'ps-privat24',
                'way'          => '',
                'title'        => _t('bills', 'Приват24'),
                'currency_id'  => 1, # рубли
                'enabled'      => true,
            ),
        );

        if ($bBalanceUse) {
            $aData = array(
                    self::PS_BALANCE => array(
                        'id'           => self::PS_UNKNOWN,
                        'class'        => 'ps-balance',
                        'way'          => '',
                        'title'        => _t('bills', 'Счет [name]', array('name' => config::sys('site.title'))),
                        'enabled'      => true,
                    )
                ) + $aData;
        }

        # Исключаем выключенные способы оплаты
        foreach ($aData as $k=>$v) {
            if ($k !== self::PS_BALANCE && (
                     empty($v['enabled']) ||
                     ! static::i()->isPaySystemAllowed($v['id'])
                )) {
                unset($aData[$k]);
            }
        }

        return $aData;
    }

    /**
     * Формирование формы запроса к системе оплаты
     * @param int $nPaySystem система оплаты Bills::PS_
     * @param string $sPaySystemWay способ оплаты (в системе оплаты)
     * @param int $nBillID id счета
     * @param float $fAmout сумма
     * @param array $aExtra доп. информация
     */
    public function buildPayRequestForm($nPaySystem, $sPaySystemWay, $nBillID, $fAmout, array $aExtra = array())
    {
        if (in_array($nPaySystem, array(self::PS_PRIVAT24))) {
            $aExtra['template'] = 'pay.request.form';
        }

        return parent::buildPayRequestForm($nPaySystem, $sPaySystemWay, $nBillID, $fAmout, $aExtra);
    }
}