<?php

use bff\utils\TextParser;

class Afisha extends AfishaBase
{
    var $itemsTable = '';
    var $typeTemplatesDir = '';

    public function route()
    {
        $prefix = ( ! bff::subdomainsEnabled('afisha') ? 'afisha/' : '' );
        $res = bff::route(array(
            $prefix.'([a-z]+)/places/(.*)-([\d]+)(\/|)(.*)'  => 'afisha/\\1:places_view/id=\\3', # /movie/places/1.html - описание кинотеатра
            $prefix.'([a-z]+)/(.*)-([\d]+)\.html(.*)'        => 'afisha/\\1:view/id=\\3',        # /movie/1.html - описание фильма
            $prefix.'([a-z]+)/soon(\/|)'                     => 'afisha/\\1:listing/soon=1',     # /movie/soon/ - фильмы скоро
            $prefix.'([a-z]+)/archive(.*)'                   => 'afisha/\\1:listing/archive=1',  # /movie/archive/ - фильмы архив
            $prefix.'([a-z]+)/places(\/|)'                   => 'afisha/\\1:places',             # /movie/places/ - кинотеатры
            $prefix.'add\.html(.*)'                          => 'afisha/add',                    # /add.html - добавление события пользователем
            $prefix.'([a-z]+)(.*)'                           => 'afisha/\\1:listing',            # /movie/ - фильмы сегодня, ?date=yyyy-mm-dd - за указанный день
        ), true);

        if ($res['event'] === false) {
            return $this->index(); # все события
        }
        
        return $this->$res['event']();
    }

    /**
     * Блок "Афиша" на главной
     * @return string
     */
    public function indexBlock()
    {
        $aData = array('today'=>array(), 'today_start'=>0);

        # получаем краткое описание и кол-во сегодняшних событий
        $today = $this->db->select_key('SELECT GROUP_CONCAT(P.item_id) as events, P.type_id
                          FROM '.TABLE_AFISHA_PLACES.' P
                          WHERE (:today BETWEEN P.period_from AND P.period_to)
                          '.($this->regionsFilterEnabled() ? ' AND P.city_id = '.Geo::cityID() : '').'
                          GROUP BY P.type_id', 'type_id', array(':today'=>date('Y-m-d', mktime(0,0,0))));
        if ( ! empty($today))
        {
            $i = 0;
            foreach ($today as $typeID=>&$v)
            {
                $type = $this->getTypeSettings($typeID);
                $eventsID = explode(',', $v['events']); $eventsID = array_unique($eventsID);
                $v['cnt'] = sizeof($eventsID);
                if ($v['cnt'] > 15) { # урезаем до 15
                    $eventsID = array_slice($eventsID, 0, 15, true);
                }

                if (empty($eventsID)) continue;

                # получаем информацию о событиях
                $v['e'] = $this->db->select('SELECT E.id, EL.title, EL.title_alt, E.created, E.link, E.img
                        FROM '.$type->tableName().' E, '.TABLE_AFISHA_ITEMS_LANG.' EL
                        WHERE E.id IN('.join(',',$eventsID).') '.$this->model->langAnd($type->id, true, 'E', 'EL'));

                if ( ! empty($v['e'])) {
                    foreach ($v['e'] as &$e) {
                        $e['img'] = $this->getImagesPath(true, array('id'=>$e['id'], 'created'=>$e['created'],
                                                          'filename'=>$e['img'], 'size'=>'s'), $type);
                        $e['url'] = static::url('view', $e['link']);
                        $e['title'] = static::urlInTitle($e['title']);
                    } unset($e);
                }

                $v['url_list'] = static::url('list', $typeID);
                $v['active'] = !$i++;
                if ($v['active']) {
                    $aData['today_start'] = $typeID;
                }
                $v['last'] = false;

            } $v['last'] = true; unset($v);

            if ( ! empty($today)) {
                reset($today);
            }
            $aData['today'] = &$today;
        }

        $urlList = Items::url('map', array('cat'=>'otdyh-i-razvlechenija/'));
        $aData['catalog'] = array(
            'main_url' => $urlList,
            'cats' => array(
                array('t'=>_t('items', 'Кинотеатры'),'url'=>$urlList.'kinoteatry'),
                array('t'=>_t('items', 'Театры, цирки'),'url'=>$urlList.'teatry'),
                array('t'=>_t('items', 'Музеи'),'url'=>$urlList.'muzei'),
                array('t'=>_t('items', 'Ночные клубы'),'url'=>$urlList.'nochnyje-kluby'),
                array('t'=>_t('items', 'Зоны отдыха'),'url'=>$urlList.'zony-otdyha'),
                array('t'=>_t('items', 'Рестораны, бары'),'url'=>$urlList.'restorany-kafe-bary'),
                array('t'=>_t('items', 'Бильярд'),'url'=>$urlList.'biljard'),
                array('t'=>_t('items', 'Боулинг'),'url'=>$urlList.'bouling'),
                array('t'=>_t('items', 'Дворцы культуры'),'url'=>$urlList.'dvorcy-i-doma-kultury'),
            ),
        );

        $aData['url_index'] = static::url('index');

        return $this->viewPHP($aData, 'index.block');
    }
    
    /**
     * Главная страница раздела "Афиша"
     * Просмотр обзорного списка событий всех типов, за указанный день
     * @param string:'getpost' date - просматриваемая дата событий(default: now)
     */
    public function index()
    {
        $nDate = $this->prepareDate($this->input->getpost('date', TYPE_NOTAGS), $bToday, true); # => "YYYY-MM-DD"
        $sDate = date('Y-m-d', $nDate);
        
        $aEvents = array();
        $aEventsCnt = $this->db->select('SELECT GROUP_CONCAT(P.item_id) as events, P.type_id
                          FROM '.TABLE_AFISHA_PLACES.' P
                          WHERE (:date BETWEEN P.period_from AND P.period_to)
                          '.($this->regionsFilterEnabled() ? ' AND P.city_id = '.Geo::cityID() : '').'
                          GROUP BY P.type_id', array(':date'=>$sDate));

        if ( ! empty($aEventsCnt))
        {
            foreach ($aEventsCnt as $v)
            {
                $typeID = $v['type_id'];
                $type = $this->getTypeSettings($typeID);
                $eventsID = explode(',', $v['events']); $eventsID = array_unique($eventsID);

                $sqlDynprops = $type->dpPrepareSelectFieldsQuery('E.', false);
                
                if ($typeID == self::TYPE_MOVIE)
                {
                    # event data
                    $aEvent = $this->db->one_array('SELECT E.id, EL.title, EL.title_alt, E.created, E.video, E.link, E.img, EL.content_short '.$sqlDynprops.'
                            FROM '.$type->tableName().' E,
                                 '.TABLE_AFISHA_ITEMS_LANG.' EL
                            WHERE E.id IN('.join(',', $eventsID).')
                                AND E.enabled = 1 AND E.moderated = 1 '.$this->model->langAnd($type->id, true, 'E', 'EL').'
                                ORDER BY RAND()
                                LIMIT 1');

                    if ( ! empty($aEvent))
                    {
                        # event places
                        $aEvent['places'] = $this->db->select('SELECT P.period_time as period, P.place_id as id,
                                IL.title, I.link_short as link
                              FROM '.TABLE_AFISHA_PLACES.' P, '.TABLE_ITEMS.' I, '.TABLE_ITEMS_LANG.' IL
                              WHERE P.type_id = :type AND P.item_id = :id
                                    AND (:date BETWEEN P.period_from AND P.period_to)
                                    AND P.place_id = I.id
                                    AND P.place_id = IL.id AND IL.lang = :lng
                              ORDER BY IL.title',
                              array(':type'=>$typeID, ':id'=>$aEvent['id'], ':date'=>$sDate, ':lng'=>LNG));
                        if ( ! empty($aEvent['places'])) {
                            foreach ($aEvent['places'] as &$p) {
                                $p['link'] = static::url('place', array('type'=>$typeID, 'link'=>$p['link']));
                            } unset($p);
                        }
                    }
                } 
                else 
                {
                    # event data, place data, time data
                    $sqlDynprops = $type->dpPrepareSelectFieldsQuery('E.', true);
                    $aEvent = $this->db->one_array('SELECT E.id, EL.title, EL.title_alt, E.created, E.video, E.link, E.img, EL.content_short '.$sqlDynprops.',
                                 P.period_time as period, P.place_id,
                                 IF(P.place_id > 0, IL.title, P.place_title) as place_title, I.link_short as place_link
                            FROM '.$type->tableName().' E,
                                 '.TABLE_AFISHA_ITEMS_LANG.' EL,
                                 '.TABLE_AFISHA_PLACES.' P
                                LEFT JOIN '.TABLE_ITEMS.' I ON P.place_id = I.id
                                LEFT JOIN '.TABLE_ITEMS_LANG.' IL ON P.place_id = IL.id AND IL.lang = :lng
                            WHERE E.id IN('.join(',', $eventsID).') AND E.enabled = 1 AND E.moderated = 1
                              AND E.id = P.item_id AND P.type_id = :type
                              AND (:date BETWEEN P.period_from AND P.period_to) '.
                    $this->model->langAnd($type->id, true, 'E', 'EL').'
                            ORDER BY RAND()
                            LIMIT 1', array(':type'=>$typeID, ':date'=>$sDate, ':lng'=>LNG));
                    if ( ! empty($aEvent)) {
                        if ($aEvent['place_id'] > 0) {
                            $aEvent['place_link'] = static::url('place', array('type'=>$typeID, 'link'=>$aEvent['place_link']));
                        }
                    }
                }
                
                if (empty($aEvent)) continue;
                
                $aEvent['img'] = $this->getImagesPath(true, array('id'=>$aEvent['id'], 'created'=>$aEvent['created'], 'filename'=>$aEvent['img'], 'size'=>'n'), $type);
                $aEvent['link'] = static::url('view', $aEvent['link']);
                $aEvent['title'] = static::urlInTitle($aEvent['title']);

                $aEvents[$typeID] = array(
                    'cnt'  => sizeof($eventsID),
                    'many' => ($type->placesMany()?1:0),
                    'e'    => $aEvent,
                    'url_list' => static::url('list', $typeID),
                );
            }
        }
        
        $aData = array(
            'events' => &$aEvents,
            'date'   => ( !$bToday ? '?date='.$sDate : '' ),
            'days'   => $this->prepareDaysFromToday($nDate, static::url('index', array('date'=>''))),
        );

        # SEO: Главная страница
        $this->urlCorrection(static::url('index'));
        $this->seo()->canonicalUrl(static::url('index', array(), true), array('date'=>$sDate));
        $this->setMeta('index', array(
            'region' => Geo::filter('title'),
            'date'   => tpl::date_format2($nDate, false, false, ' ', ', ', true),
            'date2'  => tpl::date_format2($nDate, false, false),
            'date3'  => tpl::dateFormat($nDate),
        ), $aData);

        bff::setActiveMenu('//afisha/all');
        $this->initRightblock(0, __FUNCTION__);
        return $this->viewPHP($aData, 'index');
    }
    
    /**
     * Просмотр списка событий, определенного типа, за указанный день
     *   boolean::'soon' - список событий, которые произойдут "скоро"
     *   boolean::'archive' - архив списка событий,
     * @param int $typeID тип
     * @param string $typeKey ключ типа
     */
    public function listing($typeID, $typeKey)
    {
        $this->initRightblock($typeID, __FUNCTION__);
        if ($typeID == self::TYPE_MOVIE)
        {
            if ( ! empty($_GET['soon'])) {
                return $this->listing_soon($typeID, $typeKey);
            } else if ( ! empty($_GET['archive'])) {
                return $this->listing_archive($typeID, $typeKey);
            } else {
                return $this->listing_day($typeID, $typeKey);
            }
        }

        return $this->listing_period($typeID, $typeKey);
    }
    
    /**
     * Просмотр списка событий, определенного типа, за указанный день
     *   string::'date' - просматриваемая дата событий(null|wrong => now)
     * @param int $typeID тип
     * @param string $typeKey ключ типа
     */
    public function listing_day($typeID, $typeKey)
    {
        $nDate = $this->prepareDate( $this->input->getpost('date', TYPE_NOTAGS), $bToday, true ); // => "YYYY-MM-DD"

        $aEvents = array();
        $aEventsPlaces = $this->db->select('SELECT P.item_id as event_id, P.period_time, P.place_id, IL.title as place_title, I.link_short as link
                          FROM '.TABLE_AFISHA_PLACES.' P, '.TABLE_ITEMS.' I, '.TABLE_ITEMS_LANG.' IL, '.$this->itemsTable.' E
                          WHERE P.type_id = :type
                                '.($this->regionsFilterEnabled() ? ' AND P.city_id = '.Geo::cityID() : '').'
                                AND (:date BETWEEN P.period_from AND P.period_to)
                                AND P.place_id = I.id AND I.enabled = 1 AND I.moderated = 1 '.$this->db->langAnd(true, 'I','IL').'
                                AND P.item_id = E.id AND E.enabled = 1 AND E.moderated = 1
                          ORDER BY IL.title', array(':type'=>$typeID, ':date'=>date('Y-m-d', $nDate)));

        if ( ! empty($aEventsPlaces))
        {
            $timeNow = intval( date('Hi') ); //часыминуты
            $availIndexDef = ($bToday ? 0 : ( $nDate > time() ? 0 : 1000 ));
            foreach ($aEventsPlaces as $v)
            {
                if ( ! isset($aEvents[$v['event_id']])) {
                    $aEvents[$v['event_id']] = array();
                }
                
                $availIndex = $availIndexDef;
                // проверяем доступность времени события только в текущий день
                // дни ранее текущего - все время недоступно
                // дни позже текущего - все время доступно
                if ($bToday)
                {
                    // помечаем стартовый индекс доступного на текущий момент времени
                    // например, события проходит в 12:30,19:00,22:00, сейчас 15:00, значит индекс доступного времени = 1 т.е. с 19:00 и позже
                    $time = explode(',', str_replace(':','',$v['period_time'])); //11:30,12:00,12:01,02:00 => 1130,1200,1201,0200
                    $navailIndex = 0;
                    foreach ($time as $i=>$t)
                    {
                        # все что ранее 06:00 утра, считаем следующим днем, следовательно, помечаем как доступное
                        if ($t>$timeNow || $t<600) {
                            $availIndex = $i; 
                            break;
                        } else {
                            $navailIndex++;
                        }
                    }
                    if ( ! $availIndex && ($navailIndex > $availIndex)) {
                        $availIndex = $navailIndex + 1;
                    }
                }
                $aEvents[$v['event_id']][$v['place_id']] = array(
                    't'=>explode(',', $v['period_time']),
                    'a'=>$availIndex, 'title'=>$v['place_title'],
                    'link' => static::url('place', array('type'=>$typeID, 'link'=>$v['link'])));
            }

            if ( ! empty($aEvents))
            {
                $sqlDynprops = $this->type->dpPrepareSelectFieldsQuery('I.', false);
                $aEventsData = $this->db->select('SELECT I.id, IL.title, IL.title_alt, I.created, I.video, I.link, I.img, IL.content_short '.$sqlDynprops.'
                        FROM '.$this->itemsTable.' I, '.TABLE_AFISHA_ITEMS_LANG.' IL
                        WHERE I.id IN('.join(',', array_keys($aEvents)).')
                            AND I.enabled = 1 AND I.moderated = 1 '.$this->model->langAnd($typeID, true, 'I', 'IL'));
                
                if ( ! empty($aEventsData))
                {
                    foreach ($aEventsData as $v)
                    {
                        $eventID = $v['id'];
                        $v['img'] = $this->getImagesPath(true, array('id'=>$eventID, 'created'=>$v['created'], 'filename'=>$v['img'], 'size'=>'n'));
                        $v['places'] = $aEvents[$eventID];
                        $v['link'] = static::url('view', $v['link']);
                        $v['title'] = static::urlInTitle($v['title']);
                        $aEvents[$eventID] = $v;
                    }
                    
                    foreach ($aEvents as $k=>$v) {
                        # чистим время, к которому не прикрепили описание события
                        if (empty($v['id'])) {
                            unset( $aEvents[$k] );
                        }
                    }
                    
                    # сортируем массив по названию событий
                    $aEvents = func::array_subsort($aEvents, 'title', SORT_ASC);
                } else {
                    # не нашли событий
                    $aEvents = array();
                }
            }
        }
        $aData['events'] = &$aEvents;
        $aData['menu'] = $this->prepareMenu('events', false);
        $aData['days'] = $this->prepareDaysFromToday($nDate, static::url('list', array('type'=>$typeID, 'date'=>'')));

        # SEO: Список фильмов: за день
        $this->urlCorrection(static::url('list', array('type'=>$typeID)));
        $this->seo()->canonicalUrl(static::url('list', array('type'=>$typeID), true), array('date'=>date('Y-m-d', $nDate)));
        $this->setMeta('list-day-'.$typeID, array(
            'region' => Geo::filter('title'),
            'date' => tpl::date_format2($nDate, false, false, ' ', ', ', true),
            'date2' => tpl::date_format2($nDate, false, false),
            'date3' => tpl::dateFormat($nDate),
        ), $aData);

        bff::setActiveMenu('//afisha/'.$typeKey);
        return $this->viewPHP($aData, 'listing.day', $this->typeTemplatesDir);
    }
    
    /**
     * Просмотр списка событий, определенного типа, за промежуток времени
     *   integer::'next' - кол-во пропускаемых периодов, 0 - ближайший период от текущей даты
     * @param int $typeID тип
     * @param string $typeKey ключ типа
     */
    public function listing_period($typeID, $typeKey)
    {
        # подготавливаем период просмотра
        $nOffset = $this->input->getpost('next', TYPE_UINT);
        if ($nOffset > 6) $nOffset = 0; # для просмотра допустимо не более 6 периодов вперед
        $now = mktime(0,0,0);
        $nPeriodStart = ( $nOffset ? strtotime('+ '.(($nOffset*30) + $nOffset).' days', $now) : $now );
        $nPeriodFinish = strtotime('+ 30 days', $nPeriodStart);
        
        $aEvents = array();
        $aEventsDates = $this->db->select('SELECT P.item_id as event_id, P.period_from as p_from, P.period_to as p_to, P.period_time as p_time
                          FROM '.TABLE_AFISHA_PLACES.' P, '.$this->itemsTable.' E
                          WHERE P.type_id = :type
                            '.($this->regionsFilterEnabled() ? ' AND P.city_id = '.Geo::cityID() : '').'
                            AND (
                              (P.period_from BETWEEN :start AND :finish) OR
                              (P.period_to BETWEEN :start AND :finish) OR
                              (:start BETWEEN P.period_from AND P.period_to) OR
                              (:finish BETWEEN P.period_from AND P.period_to)
                            )
                            AND P.item_id = E.id AND E.enabled = 1 AND E.moderated = 1
                          ORDER BY P.period_from ASC', array(
                            ':type'   => $typeID,
                            ':start'  => date('Y-m-d', $nPeriodStart),
                            ':finish' => date('Y-m-d', $nPeriodFinish),
                          ));

        if ( ! empty($aEventsDates))
        {
            $months = $this->locale->getMonthTitle();
            $days = explode(';', _t('', 'воскресенье;понедельник;вторник;среда;четверг;пятница;суббота'));
            
            # формируем список активных дней и связываем их с событиями
            $aDays = array();
            
            $daysSpecial = array($now => _t('','Сегодня'), ($now + 86400) => _t('','Завтра'), ($now + 172800)=>_t('','Послезавтра'));
            foreach ($aEventsDates as $v)
            {
                list($f['y'], $f['m'], $f['d']) = explode('-', $v['p_from'], 3); // от
                list($t['y'], $t['m'], $t['d']) = explode('-', $v['p_to'], 3); // до

                $fn = mktime(0,0,0,$f['m'],$f['d'],$f['y']);
                $tn = mktime(0,0,0,$t['m'],$t['d'],$t['y']);

                $v['p_time'] = ($v['p_time']!='' ? ', '. $v['p_time'] : ''); // , 12:00

                $aEvents[$v['event_id']][] = (($f['d'].' '.$months[(int)$f['m']] . ( $tn > $fn ? ' - '.$t['d'].' '.$months[(int)$t['m']] : '' ) ) . $v['p_time']);
                
                $nDay = $nPeriodStart;
                while($nDay <= $nPeriodFinish) 
                {
                    if ($nDay >= $fn && $nDay <= $tn) {
                        if ( ! isset($aDays[$nDay])) {
                            list($d,$m,$wd) = explode(',', date('j,n,w', $nDay));
                            $aDays[$nDay] = array( 's0'=>( isset($daysSpecial[$nDay]) ? $daysSpecial[$nDay].', ' : ''), // Сегодня, 
                                        's1'=> $d.' '.$months[$m], // 12 января
                                        's2'=> ' (' . $days[$wd].')', // (среда)
                                        't'=>($now == $nDay), // сегодня?
                                        'e'=>array() ); // события
                        }
                        $aDays[$nDay]['e'][$v['event_id']] = $v['p_time']; // связываем день с событием, + сохраняем время события в этот день
                    }
                    $nDay += 86400;
                }
            }

            if ( ! empty($aEvents))
            {
                $sqlDynprops = $this->type->dpPrepareSelectFieldsQuery('E.', false);
                $aEventsData = $this->db->select('SELECT E.id, EL.title, EL.title_alt, E.created, E.video, E.link, E.img, EL.content_short '.$sqlDynprops.',
                            P.place_id, IF(P.place_id > 0, IL.title, P.place_title) as place_title, I.link_short as place_link
                        FROM '.$this->itemsTable.' E,
                            '.TABLE_AFISHA_ITEMS_LANG.' EL,
                            '.TABLE_AFISHA_PLACES.' P
                            LEFT JOIN '.TABLE_ITEMS.' I ON P.place_id = I.id AND I.enabled = 1 AND I.moderated = 1
                            LEFT JOIN '.TABLE_ITEMS_LANG.' IL ON '.$this->db->langAnd(false, 'I','IL').'
                        WHERE E.id IN('.join(',', array_keys($aEvents)).')
                            AND E.enabled = 1 AND E.moderated = 1
                            AND E.id = P.item_id 
                            AND P.type_id = '.$typeID.
                            $this->model->langAnd($typeID, true, 'E', 'EL').'
                        GROUP BY E.id');

                if ( ! empty($aEventsData))
                {
                    foreach ($aEventsData as $v)
                    {
                        $eventID = $v['id'];
                        $v['img'] = $this->getImagesPath(true, array('id'=>$eventID, 'created'=>$v['created'], 'filename'=>$v['img'], 'size'=>'n'));
                        $v['h'] = 0;
                        $v['periods'] = $aEvents[$eventID];
                        $v['link'] = static::url('view', $v['link']);
                        $v['title'] = static::urlInTitle($v['title']);
                        if ($v['place_id'] > 0) {
                            $v['place_link'] = static::url('place', array('type'=>$typeID, 'link'=>$v['place_link']));
                        }
                        $aEvents[$eventID] = $v;
                    } unset($aEventsData);

                    foreach ($aEvents as $k=>$v) {
                        # пропускаем время, к которому не прикрепили описание события
                        if (empty($v['id'])) unset($aEvents[$k]);
                    }

                } else {
                    # не нашли событий
                    $aEvents = array();
                }
            }
        }
        $aData['days'] = &$aDays;
        $aData['events'] = &$aEvents;
        $aData['menu'] = $this->prepareMenu('events', false);
        $aData['period'] = $this->prepareDaysPeriod($nPeriodStart, $nPeriodFinish, $nOffset, static::url('list', array('type'=>$typeID, 'next'=>'')));

        # SEO: Список событий: период
        $this->urlCorrection(static::url('list', array('type'=>$typeID)));
        $this->seo()->canonicalUrl(static::url('list', array('type'=>$typeID), true), array('next'=>$nOffset));
        $this->setMeta('list-period-'.$typeID, array(
            'region' => Geo::filter('title'),
            'period-from' => tpl::date_format2($nPeriodStart, false, true, ' ', ', ', true),
            'period-to' => tpl::date_format2($nPeriodFinish, false, false),
        ), $aData);

        bff::setActiveMenu('//afisha/'.$typeKey);
        return $this->viewPHP($aData, 'listing.period');
    }
    
    /**
     * Просмотр списка событий, определенного типа, которые произойдут "скоро"
     * Типы: movie
     * @param int $typeID тип
     * @param string $typeKey ключ типа
     */
    public function listing_soon($typeID, $typeKey)
    {
        $sqlDynprops = $this->type->dpPrepareSelectFieldsQuery('E.', false);
        $aEvents = $this->db->select('SELECT E.id, EL.title, EL.title_alt, E.created, E.video, E.link, E.img, EL.content_short '.$sqlDynprops.'
                FROM '.$this->itemsTable.' E
                    LEFT JOIN '.TABLE_AFISHA_PLACES.' PT ON PT.type_id = :type AND PT.item_id = E.id
                        AND (:now BETWEEN PT.period_from AND PT.period_to),
                    '.TABLE_AFISHA_ITEMS_LANG.' EL,
                    '.TABLE_AFISHA_PLACES.' P
                WHERE PT.item_id IS NULL 
                    AND E.enabled = 1 AND E.moderated = 1
                    AND P.type_id = :type
                    '.($this->regionsFilterEnabled() ? ' AND P.city_id = '.Geo::cityID() : '').'
                    AND P.item_id = E.id AND P.period_from >= :now'.
                    $this->model->langAnd($typeID, true, 'E', 'EL').'
                GROUP BY E.id
                ORDER BY EL.title', array(
                    ':type' => $typeID,
                    ':now'  => date('Y-m-d', mktime(0,0,0)),
                ));
        
        if ( ! empty($aEvents))
        {
            foreach ($aEvents as &$v) {
                $v['img'] = $this->getImagesPath(true, array('id'=>$v['id'], 'created'=>$v['created'], 'filename'=>$v['img'], 'size'=>'n'));
                $v['link'] = static::url('view', $v['link']);
                $v['title'] = static::urlInTitle($v['title']);
            } unset($v);
        }
        $aData['events'] = &$aEvents;
        $aData['menu'] = $this->prepareMenu('soon', false);

        # SEO: Список фильмов: скоро
        $this->urlCorrection(static::url('list.tab', array('type'=>$typeID, 'tab'=>'soon')));
        $this->seo()->canonicalUrl(static::url('list.tab', array('type'=>$typeID, 'tab'=>'soon'), true));
        $this->setMeta('list-soon-'.$typeID, array('region' => Geo::filter('title')), $aData);

        bff::setActiveMenu('//afisha/'.$typeKey);
        return $this->viewPHP($aData, 'listing.soon', $this->typeTemplatesDir);
    }

    /**
     * Просмотр списка архива событий, определенного типа
     * Типы: movie
     * @param int $typeID тип
     * @param string $typeKey ключ типа
     */
    public function listing_archive($typeID, $typeKey)
    {
        $f = $this->input->getm(array(
            'l' => TYPE_NOTAGS, # буква
            'g' => TYPE_UINT,   # жанр
            'q' => TYPE_NOTAGS, # поиск строки
        ));
        
        $sql = array();
        $sql[] = 'E.enabled = 1';   
        $sql[] = 'E.moderated = 1';
        if ($this->regionsFilterEnabled()) $sql[] = 'E.city_id = '.Geo::cityID();
        $pagesLink = static::url('list.tab', array('type'=>$typeID, 'tab'=>'archive')).'?page={page}';
        $search = false;
        $seoIndex = true;
        if (!empty($f['l'])) {
            if ($f['l'] == 'num') {
                $letter = $this->db->str2sql('[[:digit:]]');
                $sql[] = '(EL.title REGEXP '.$letter.' OR E.f5 REGEXP '.$letter.')';  # ищем по первой цифре в названии, в оригинальном названии
                // $search = 'Буква: '.$f['l'];
            } else {
                $letter = $this->db->str2sql($f['l'].'%');
                $sql[] = '(EL.title LIKE '.$letter.' OR E.f5 LIKE '.$letter.')';  # ищем по первой букве названия, оригинального названия
                $search = _t('afisha', 'Фильмы на букву: [letter]', array('letter'=>$f['l']));
            }
            $pagesLink .= '&l='.$f['l'];
            $seoIndex = false;
        } elseif ($f['g'] > 0) {
            if ( ! empty($this->type->dp[static::MOVIE_GENRE_DP_ID]['multi'])) {
                foreach ($this->type->dp[static::MOVIE_GENRE_DP_ID]['multi'] as $v) {
                    if ($v['value'] == $f['g']) {
                        $sql[] = '(E.f1 & '.$f['g'].')'; # ищем по жанру
                        $search = _t('afisha', 'Жанр: [genre]', array('genre'=>$v['name']));
                        $pagesLink .= '&g='.$f['g'];
                        $seoIndex = false;
                    }
                }
            }
        } elseif (!empty($f['q']) && mb_strlen($f['q']) >=3) {
            $query = $this->db->str2sql('%'.$f['q'].'%');
            $sql[] = '(EL.title LIKE '.$query.')'; # ищем по названию
            $search = _t('afisha', 'По названию: "[title]"', array('title'=>HTML::escape($f['q'])));
            $pagesLink .= '&q='.urlencode($f['q']);
            $seoIndex = false;
        }
            
        $sql = join(' AND ', $sql);

        $nPage = $this->input->get('page', TYPE_UINT); if ( ! $nPage) $nPage = 1;
        $nPerpage = 10;
        $nTotal = $this->db->one_data('SELECT COUNT(*) FROM '.$this->itemsTable.' E, '.TABLE_AFISHA_ITEMS_LANG.' EL WHERE '.$sql.$this->model->langAnd($typeID, true, 'E', 'EL'));
        $aData['pgn'] = $this->generatePagenationDots($nTotal, $nPerpage, 2, $pagesLink, $sqlLimit);
        
        $sqlDynprops = $this->type->dpPrepareSelectFieldsQuery('E.', false);
        $aEvents = $this->db->select('SELECT E.id, EL.title, EL.title_alt, E.created, E.video, E.link, E.img, EL.content_short '.$sqlDynprops.'
                FROM '.$this->itemsTable.' E,
                     '.TABLE_AFISHA_ITEMS_LANG.' EL
                WHERE '.$sql.$this->model->langAnd($typeID, true, 'E', 'EL').'
                ORDER BY EL.title'.$sqlLimit);
        
        if ( ! empty($aEvents))
        {
            foreach ($aEvents as &$v) {
                $v['img'] = $this->getImagesPath(true, array('id'=>$v['id'], 'created'=>$v['created'], 'filename'=>$v['img'], 'size'=>'n'));
                $v['link'] = static::url('view', $v['link']);
                $v['title'] = static::urlInTitle($v['title']);
            } unset($v);
        }

        $aData['events'] = &$aEvents;
        $aData['menu'] = $this->prepareMenu('archive', ($search!==false || $nPage >1));
        $aData['page'] = $nPage;
        $aData['search'] = $search;

        # SEO: Список фильмов: архив
        $this->urlCorrection(static::url('list.tab', array('type'=>$typeID, 'tab'=>'archive')));
        $this->seo()->robotsIndex($seoIndex);
        $this->seo()->canonicalUrl(static::url('list.tab', array('type'=>$typeID, 'tab'=>'archive'), true), array(
            'page' => $nPage
        ));
        $this->setMeta('list-archive-'.$typeID, array('region' => Geo::filter('title'), 'page'=>$nPage), $aData);

        bff::setActiveMenu('//afisha/'.$typeKey);
        return $this->viewPHP($aData, 'listing.archive', $this->typeTemplatesDir);
    }
    
    /**
     * Добавление события пользователем
     */
    public function add()
    {
        if ( ! empty($_GET['success'])) {
            bff::setActiveMenu('//afisha/all', true);
            bff::setMeta(_t('afisha','Новое событие'));
            return $this->showSuccess(_t('afisha','Новое событие'), _t('afisha','Событие было успешно добавлено, после проверки модератором,[br]оно будет опубликовано.', array('br' => '<br />')));
        }
        
        $aData = $this->input->postm(array(
            'type'      => TYPE_UINT,   # Тип события
            'title'     => TYPE_NOTAGS, # Заголовок
            'content'   => array(TYPE_NOTAGS, 'len'=>500), # Описание (500 символов макс.)
            'place'     => TYPE_NOTAGS, # Место
            'place_city'    => TYPE_UINT,   # Город
            'place_address' => TYPE_NOTAGS, # Адрес
            'place_phone'   => TYPE_NOTAGS, # Телефон
            'user_type' => TYPE_UINT,   # Тип пользователя: организатор / пользователь
            'name'      => TYPE_NOTAGS, # Имя
            'phone'     => TYPE_NOTAGS, # Телефон
            'email'     => TYPE_NOTAGS, # E-mail
        ));

        if(Request::isPOST()) {
            $aResp = array();
            do {

                if (!User::id()) {
                    $this->errors->set(_t('afisha', 'Возможность добавлять информацию о событиях доступна только для авторизованных пользователей'));
                    break;
                }

                $aTypeData = $this->db->one_array('
                SELECT id, title_' . LNG . ' AS title FROM ' . TABLE_AFISHA_TYPES . ' WHERE id = ' . $aData['type']);
                if (empty($aTypeData)) {
                    $this->errors->set(_t('afisha', 'Тип события указан некорректно'));
                } else {
                    $this->type = $this->getTypeSettings($aData['type']);
                }

                if (empty($aData['title'])) {
                    $this->errors->set(_t('afisha', 'Укажите заголовок'));
                }

                $aData['content'] = $this->input->cleanTextPlain($aData['content'], false, false);
                if (empty($aData['content'])) {
                    $this->errors->set(_t('afisha', 'Укажите описание события'));
                }

                if (!Geo::cityIsValid($aData['place_city'])) {
                    $this->errors->set(_t('afisha', 'Выберите город из доступных в списке'));
                }

                # пользователь
                if (!in_array($aData['user_type'], array(self::USERTYPE_ORGANIZER, self::USERTYPE_MEMBER))) {
                    $aData['user_type'] = self::USERTYPE_MEMBER;
                }

                if ($aData['user_type'] == self::USERTYPE_MEMBER) {
                    $aData['user_info'] = array('name' => '', 'phone' => '', 'email' => '');
                } else {
                    $aData['user_info'] = array('name' => $aData['name'], 'phone' => $aData['phone'], 'email' => $aData['email']);
                    if (empty($aData['name'])) {
                        $this->errors->set(_t('afisha', 'Укажите ваше имя'));
                    }
                    if (empty($aData['phone'])) {
                        if (empty($aData['email'])) {
                            $this->errors->set(_t('afisha', 'Укажите ваш номер телефона'));
                        } else {
                            if (!$this->input->isEmail($aData['email'], false)) {
                                $this->errors->set(_t('afisha', 'Укажите корректный e-mail адрес'));
                            }
                        }
                    }
                }

                if (!$this->errors->no()) break;

                if (!$this->security->validateToken()) {
                    $this->errors->reloadPage();
                } else {
                    $nEventID = $this->db->insert($this->type->tableName(), array(
                        'user_id' => User::id(),
                        'user_type' => $aData['user_type'],
                        'user_info' => serialize($aData['user_info']),
                        'country_id' => Geo::defaultCountry(),
                        'city_id' => $aData['place_city'],
                        'created' => $this->db->now(),
                        'moderated' => 0,
                    ), 'id');

                    if ($nEventID > 0) {
                        $this->db->langInsert(array('id' => $nEventID, 'type_id' => $this->type->id), array(
                            'title' => array(LNG => $aData['title']),
                            'content_short' => array(LNG => $aData['content'] . "\nМесто: {$aData['place']}\nГород: " . Geo::regionTitle($aData['place_city']) . "\nАдрес: {$aData['place_address']}\nТелефон: {$aData['place_phone']}"),
                        ), $this->model->langItems, TABLE_AFISHA_ITEMS_LANG);

                        $this->itemsModerationCounter($this->type->keyword, 1);

                        # загружаем изображение
                        $oImages = $this->initImages(array('id' => $nEventID, 'created' => $this->db->now()), $this->type);
                        $oImages->upload('img');

                        # формируем ссылку в заголовке
                        $aData['link'] = '';
                        $aData['title'] = array(LNG => $aData['title']);
                        foreach ($this->locale->getLanguages() as $l) {
                            if ($l == LNG) continue;
                            $aData['title'][$l] = '';
                        }
                        $aData['city_id'] = $aData['place_city'];

                        if ($this->prepareItemTitle($nEventID, $aData)) {
                            $this->model->itemSave($nEventID, array(
                                'link' => $aData['link'],
                                'title' => $aData['title'],
                                'title_params' => $aData['title_params'],
                            ), $this->type->id
                            );
                        }

                        $aResp['redirect'] = static::url('add', array('success' => 1));
                    }
                }
            } while (false);
            $this->iframeResponseForm($aResp);
        }

        $aData['hash'] = $this->security->getToken();
        $aData['types'] = $this->db->select('SELECT id, title_'.LNG.' AS title FROM '.TABLE_AFISHA_TYPES.' WHERE enabled = 1 ORDER BY num');

        # SEO: Добавление события
        $this->urlCorrection(static::url('add'));
        $this->seo()->canonicalUrl(static::url('add', array(), true));
        $this->setMeta('add', array('region' => Geo::filter('title')), $aData);

        bff::setActiveMenu('//afisha/all', true);
        return $this->viewPHP($aData, 'add');
    }
    
    /**
     * Просмотр события
     *   integer::'id' - id события
     *   string::'date' - просматриваемая дата рассписания события (в случае с кино и подобными)
     * @param int $typeID тип
     * @param string $typeKey ключ типа
     */
    public function view($typeID, $typeKey)
    {
        $nEventID = $this->input->get('id', TYPE_UINT);

        if (Request::isAJAX())
        {
            $sAction = $this->input->get('act');
            if ( ! $nEventID) {
                $sAction = '?';
            }

            $aResponse = array();
            switch ($sAction)
            {
                case 'comment-add':
                {
                    if ( ! $this->security->validateToken() ) {
                        $this->errors->reloadPage(); break;
                    }

                    $nParentCommentID = $this->input->post('reply', TYPE_UINT);
                    $sMessage = $this->input->post('message', TYPE_STR);
                    $oTextParser = new TextParser();
                    $sMessage = $oTextParser->parseCommentPlain($sMessage, 3000);
                    if (strlen($sMessage) < 5) {
                        $this->errors->set(_t('afisha', 'Минимальная длина текста комментария: 5 символов')); break;
                    }

                    $oComments = $this->itemComments($typeID);
                    $nCommentID = $oComments->commentInsert($nEventID, array('message' => $sMessage), $nParentCommentID);
                    $aResponse['comment_id'] = $nCommentID;
                    $aResponse['total'] = $oComments->commentsTotal($nEventID);
                } break;
                case 'comment-response':
                {
                    $nCommentLastID = $this->input->post('comment_id_last', TYPE_UINT);
                    $oComments = $this->itemComments($typeID);
                    $aResponse = $oComments->commentsDataFront($nEventID, $nCommentLastID);
                    $aResponse['aComments'] = tpl::commentsBlock($aResponse['aComments'], true, $oComments);
                } break;
                default:{
                    $this->errors->reloadPage();
                } break;
            }
            
            $this->ajaxResponseForm($aResponse);
        }
        
        $sTypeMenuActive = 'events';
        
        if ($typeID == self::TYPE_MOVIE)
        {
            # просмотр события, с множественным связыванием с объектами
            # - только события типа self::TYPE_MOVIE
            $sqlDynprops = $this->type->dpPrepareSelectFieldsQuery('E.', true);
            $aData = $this->db->one_array('SELECT E.id, E.created, E.link, E.img, E.video, E.video_data,
                        EL.title, EL.title_alt, E.content, EL.content_short,
                        EL.mtitle, EL.mkeywords, EL.mdescription, E.mtemplate, EL.share_title, EL.share_description, EL.share_sitename
                        '.$sqlDynprops.'
                    FROM '.$this->itemsTable.' E,
                         '.TABLE_AFISHA_ITEMS_LANG.' EL
                    WHERE E.id = '.$nEventID.' AND E.enabled = 1 AND E.moderated = 1 '.$this->model->langAnd($typeID, true, 'E', 'EL'));

            if (empty($aData)) {
                $this->errors->error404();
            }
            $nDate = $this->prepareDate( $this->input->getpost('date', TYPE_NOTAGS), $bToday, true ); // => "YYYY-MM-DD"
            $aData['title'] = static::urlInTitle($aData['title']);
            $aData['places'] = array(); 
            $aEventPlaces = $this->db->select('SELECT P.item_id as event_id, P.period_time, P.place_id,
                                    IL.title as place_title, I.link_short as place_link
                              FROM '.TABLE_AFISHA_PLACES.' P, '.TABLE_ITEMS.' I, '.TABLE_ITEMS_LANG.' IL
                              WHERE P.type_id = :type AND P.item_id = :event
                                    AND (:date BETWEEN P.period_from AND P.period_to)
                                    AND P.place_id = I.id AND I.enabled = 1 AND I.moderated = 1 '.$this->db->langAnd(true, 'I','IL').'
                              ORDER BY IL.title', array(':type'=>$typeID, ':event'=>$nEventID, ':date'=>date('Y-m-d', $nDate)));

            if (empty($aEventPlaces))
            {
                if ($bToday) {
                    $sTypeMenuActive = 'soon';
                }
            } else {
                $timeNow = intval( date('Hi') ); # часыминуты
                $availIndexDef = ($bToday ? 0 : ( $nDate > time() ? 0 : 1000 ));
                foreach ($aEventPlaces as $v)
                {
                    $availIndex = $availIndexDef;
                    # проверяем доступность времени события только в текущий день
                    # дни ранее текущего - все время недоступно
                    # дни позже текущего - все время доступно
                    if ($bToday)
                    {
                        # помечаем стартовый индекс доступного на текущий момент времени
                        # например, события проходит в 12:30,19:00,22:00, сейчас 15:00, значит индекс доступного времени = 1 т.е. с 19:00 и позже
                        $time = explode(',', str_replace(':','',$v['period_time'])); //11:30,12:00,12:01,02:00 => 1130,1200,1201,0200
                        $navailIndex = 0;
                        foreach ($time as $i=>$t)
                        {
                            # все что ранее 06:00 утра, считаем следующим днем, следовательно, помечаем как доступное
                            if ($t>$timeNow || $t<600) {
                                $availIndex = $i; 
                                break;
                            } else {
                                $navailIndex++;
                            }
                        }
                        if ( ! $availIndex && ($navailIndex > $availIndex)) {
                            $availIndex = $navailIndex + 1;
                        }
                    }
                    
                    $aData['places'][$v['place_id']] = array(
                        't'     => explode(',', $v['period_time']),
                        'a'     => $availIndex,
                        'title' => $v['place_title'],
                        'link'  => static::url('place', array('type'=>$typeID, 'link'=>$v['place_link'])) );
                } unset($aEventPlaces);
            }

            $aData['img'] = $this->getImagesPath(true, array('id'=>$nEventID, 'created'=>$aData['created'], 'filename'=>$aData['img'], 'size'=>'n'));
            $aData['days'] = $this->prepareDaysFromToday($nDate, static::urlDynamic($aData['link']).'?date=', '#time');
        } 
        else
        {
            # просмотр события, с единичным связыванием с объектами
            # - все типы событий, кроме self::TYPE_MOVIE
            $sqlDynprops = $this->type->dpPrepareSelectFieldsQuery('E.', true);
            $aData = $this->db->one_array('SELECT E.id, EL.title, EL.title_alt, E.created, E.video, E.video_data, E.link,
                        E.img, EL.content_short, EL.content,
                        EL.mtitle, EL.mkeywords, EL.mdescription, E.mtemplate, EL.share_title, EL.share_description, EL.share_sitename
                        '.$sqlDynprops.',
                        P.place_id, P.place_title
                    FROM '.$this->itemsTable.' E,
                         '.TABLE_AFISHA_ITEMS_LANG.' EL,
                         '.TABLE_AFISHA_PLACES.' P
                    WHERE E.id = :event
                        AND E.enabled = 1 AND E.moderated = 1
                        AND E.id = P.item_id
                        AND P.type_id = :type'.
                        $this->model->langAnd($typeID, true, 'E', 'EL').'
                    GROUP BY E.id', array(':event'=>$nEventID, ':type'=>$typeID));
            
            if (empty($aData)) {
                $this->errors->error404();
            }

            $aData['title'] = static::urlInTitle($aData['title']);

            if ($aData['place_id'] > 0) {
                $aDataPlace = $this->db->one_array('SELECT IL.title as place_title, I.no_map, IA.phonesq as phones, IA.city_id, IA.address, I.link_short as place_link
                        FROM '.TABLE_ITEMS.' I, '.TABLE_ITEMS_LANG.' IL, '.TABLE_ITEMS_ADDR.' IA
                        WHERE I.id = '.$aData['place_id'].' AND I.enabled = 1 AND I.moderated = 1 '.$this->db->langAnd(true, 'I','IL').'
                          AND I.id = IA.item_id
                        ');
                if ( ! empty($aDataPlace)) {
                    $aData = array_merge($aData, $aDataPlace);
                    $aData['place_link'] = static::url('place', array('type' => $typeID, 'link' => $aDataPlace['place_link']));
                    $aData['address'] = Geo::regionTitle($aData['city_id']).', '.$aData['address'];
                } else {
                    $this->errors->error404();
                }
            }

            $aData['img'] = $this->getImagesPath(true, array('id'=>$nEventID, 'created'=>$aData['created'], 'filename'=>$aData['img'], 'size'=>'n'));
            
            $aEventsData = $this->db->select('SELECT P.period_from as p_from, P.period_to as p_to, P.period_time as p_time
                  FROM '.TABLE_AFISHA_PLACES.' P
                  WHERE P.type_id = :type AND P.item_id = :item
                  ORDER BY P.period_from ASC',
                  array(':type'=>$typeID, ':item'=>$nEventID));
            
            $aDates = array();
            $months = $this->locale->getMonthTitle();

            foreach ($aEventsData as $v)
            {
                list($f['y'], $f['m'], $f['d']) = explode('-', $v['p_from'], 3); # от
                list($t['y'], $t['m'], $t['d']) = explode('-', $v['p_to'], 3); # до

                $fn = mktime(0,0,0,$f['m'],$f['d'],$f['y']);
                $tn = mktime(0,0,0,$t['m'],$t['d'],$t['y']);
                
                $aDates[] = (($f['d'].' '.$months[(int)$f['m']] . ( $tn > $fn ? ' - '.$t['d'].' '.$months[(int)$t['m']] : '' ) ) . ($v['p_time']!='' ? ', '. $v['p_time'] : ''));
            }

            $aData['dates'] = join('<br />&nbsp;', $aDates);
            unset($aEventsData, $aDates);
        }

        # Подробное описание
        if ($this->type->contentPublicator()) {
            $aData['content'] = $this->initPublicator()->view($aData['content'], $nEventID, 'view.content', $this->typeTemplatesDir);
        } else if ($this->type->contentTextarea()) {
            $aData['content'] = nl2br($aData['content']);
        }

        $oComments = $this->itemComments($typeID);
        $aData['comments'] = tpl::commentsBlock($oComments->commentsDataFront($nEventID), false, $oComments);
        $aData['menu'] = $this->prepareMenu($sTypeMenuActive, true);

        # SEO: Просмотр события
        $this->urlCorrection(static::urlDynamic($aData['link']));
        $this->seo()->canonicalUrl($aData['link']);
        $this->setMeta('view', array(
            'title'       => strip_tags($aData['title']),
            'description' => tpl::truncate($aData['content_short'], 150),
            'region'      => Geo::filter('title'),
        ), $aData);
        $this->seo()->setSocialMetaOG($aData['share_title'], $aData['share_description'], array($aData['img']), $aData['link'], $aData['share_sitename']);
        $aData['link'] = static::url('view', $aData['link']);

        bff::setActiveMenu('//afisha/'.$typeKey);
        $this->initRightblock($typeID, __FUNCTION__);
        return $this->viewPHP($aData, 'view', ($typeID == self::TYPE_MOVIE ? $this->typeTemplatesDir : $this->module_dir_tpl));
    }
        
    /**
     * Просмотр списка объектов определенного типа(@example: кинотеатры, театры), в которых проходят события
     * @param int $typeID тип
     * @param string $typeKey ключ типа
     */
    public function places($typeID, $typeKey)
    {
        $places = $this->db->select('
            SELECT I.id, I.link as item_link, I.link_short, IL.title, IL.descshort as description, IA.address, IA.phonesq as phones, IA.site,
                   I.img_list as img, I.imgcnt
            FROM '.TABLE_AFISHA_PLACES.' P, '.TABLE_ITEMS.' I, '.TABLE_ITEMS_ADDR.' IA, '.TABLE_ITEMS_LANG.' IL
            WHERE P.type_id = '.$typeID.'
              AND I.id = IA.item_id '.($this->regionsFilterEnabled() ? ' AND IA.city_id = '.Geo::cityID() : '').'
              AND P.place_id!=0 AND P.place_id = I.id
              AND I.enabled = 1 AND I.moderated = 1 '.$this->db->langAnd(true, 'I','IL').'
            GROUP BY I.id
            ORDER BY I.vip DESC, IL.title
        ');

        if ( ! empty($places))
        {
            foreach ($places as &$v)
            {
                if ( ! empty($v['phones'])) {
                    $v['phones'] = join('<br/>', explode(', ', $v['phones']));
                }
                $v['place_link'] = static::url('place', array('type'=>$typeID, 'link'=>$v['link_short']));
                $v['item_link'] = Items::url('view', $v['item_link']);
            } unset($v);
        } else {
            $places = array();
        }
        $aData['places'] = &$places;
        $aData['menu'] = $this->prepareMenu('places', false);

        # SEO: Список мест
        $this->urlCorrection(static::url('places', $typeID));
        $this->seo()->canonicalUrl(static::url('places', $typeID, true));
        $this->setMeta('places-'.$typeID, array(
            'region' => Geo::filter('title'),
        ), $aData);

        bff::setActiveMenu('//afisha/'.$typeKey);
        $this->initRightblock($typeID, __FUNCTION__);
        return $this->viewPHP($aData, 'places');
    }
    
    /**
     * Просмотр объекта проведения события
     *   integer::'id' ID объекта
     * @param int $typeID тип
     * @param string $typeKey ключ типа
     */
    public function places_view($typeID, $typeKey)
    {
        $nItemID = $this->input->get('id', TYPE_UINT);
        
        $aData = $this->db->one_array('
                    SELECT I.id, I.link as item_link, I.link_short, IL.title, IL.descshort as description,
                           IA.address, IA.phonesq as phones, IA.site, I.img_list as img, I.imgcnt
                    FROM '.TABLE_ITEMS.' I, '.TABLE_ITEMS_ADDR.' IA,
                         '.TABLE_ITEMS_LANG.' IL, '.TABLE_AFISHA_PLACES.' P
                    WHERE I.id = :item AND I.enabled = 1 AND I.moderated = 1 '.$this->db->langAnd(true, 'I','IL').'
                      AND P.place_id = I.id AND I.id = IA.item_id AND P.type_id = :type
                    GROUP BY I.id
                ', array(':item'=>$nItemID, ':type'=>$typeID));

        if (empty($aData)) {
            $this->errors->error404();
        }
        
        if ( ! empty($aData['phones'])) {
            $aData['phones'] = join('<br />', explode(', ', $aData['phones']));
        }
        $aData['place_link'] = static::url('place', array('type'=>$typeID, 'link'=>$aData['link_short']));
        $aData['item_link'] = Items::url('view', $aData['item_link']);

        if ($typeID == self::TYPE_MOVIE)
        {
            $nDate = $this->prepareDate( $this->input->getpost('date', TYPE_NOTAGS), $bToday, true ); # => "YYYY-MM-DD"

            $aData['days'] = $this->prepareDaysFromToday($nDate, $aData['place_link'].'?date=');
            
            $aEvents = $this->db->select('SELECT P.item_id as event_id, EL.title, EL.title_alt, P.period_time, E.link
                              FROM '.TABLE_AFISHA_PLACES.' P,
                                   '.$this->type->tableName().' E,
                                   '.TABLE_AFISHA_ITEMS_LANG.' EL
                              WHERE P.place_id = :item AND P.type_id = :type
                                    AND (:date BETWEEN P.period_from AND P.period_to)
                                    AND P.item_id = E.id '.$this->model->langAnd($typeID, true, 'E', 'EL').'
                              GROUP BY E.id
                              ORDER BY EL.title', array(
                                ':item' => $nItemID, ':type'=>$typeID, ':date'=>date('Y-m-d', $nDate)
                              ));
            if ( ! empty($aEvents))
            {
                $timeNow = intval( date('Hi') ); //часыминуты
                $availIndexDef = ($bToday ? 0 : ( $nDate > time() ? 0 : 1000 ));
                foreach ($aEvents as $k=>$v)
                {
                    $availIndex = $availIndexDef;
                    // проверяем доступность времени события только в текущий день
                    // дни ранее текущего - все время недоступно
                    // дни позже текущего - все время доступно
                    if ($bToday)
                    {
                        // помечаем стартовый индекс доступного на текущий момент времени
                        // например, события проходит в 12:30,19:00,22:00, сейчас 15:00, значит индекс доступного времени = 1 т.е. с 19:00 и позже
                        $time = explode(',', str_replace(':','',$v['period_time'])); //11:30,12:00,12:01,02:00 => 1130,1200,1201,0200
                        $navailIndex = 0;
                        foreach ($time as $i=>$t)
                        {
                            // все что ранее 06:00 утра, считаем следующим днем, следовательно, помечаем как доступное
                            if ($t>$timeNow || $t<600) {
                                $availIndex = $i; 
                                break;
                            } else {
                                $navailIndex++;
                            }
                        }
                        if ( ! $availIndex && ($navailIndex > $availIndex)) {
                            $availIndex = $navailIndex + 1;
                        }
                    }
                    
                    $aEvents[$k]['t'] = explode(',', $v['period_time']);
                    $aEvents[$k]['a'] = $availIndex;
                    $aEvents[$k]['link'] = static::url('view', $v['link']);
                    $aEvents[$k]['title'] = static::urlInTitle($v['title']);
                }
            }
            
            $aData['events'] = &$aEvents;
        }
        $aData['menu'] = $this->prepareMenu('places', true);

        # SEO: Просмотр места события
        $this->urlCorrection($aData['place_link']);
        $this->seo()->canonicalUrl(static::url('place', array('type'=>$typeID, 'link'=>$aData['link_short']), true), array(
            'date' => ($typeID == self::TYPE_MOVIE ? date('Y-m-d', $nDate) : '')
        ));
        $this->setMeta('place-'.$typeID, array(
            'region' => Geo::filter('title'),
            'date'   => tpl::date_format2($nDate, false, false, ' ', ', ', true),
            'date2'  => tpl::date_format2($nDate, false, false),
            'date3'  => tpl::dateFormat($nDate),
        ), $aData);

        bff::setActiveMenu('//afisha/'.$typeKey);
        $this->initRightblock($typeID, __FUNCTION__);
        return $this->viewPHP($aData, 'places.view', ( $typeID == self::TYPE_MOVIE ? $this->typeTemplatesDir : $this->module_dir_tpl));
    }

    /**
     * Инициализация правого блока
     * @param int $typeID тип
     * @param string $sPage название метода, в котором выполняется инициализация правого блока
     * @param array $aExtra доп. параметры
     */
    protected function initRightblock($typeID, $sPage, $aExtra = array())
    {
        $aData = array('extra'=>'');

        switch ($sPage)
        {
            case 'listing':
            {
                if ($typeID == self::TYPE_MOVIE) {
                    $aData['extra'] = $this->viewPHP($aData, 'filter', $this->typeTemplatesDir);
                }
            } break;
        }

        bff::setRightblock( $this->viewPHP($aData, 'common.rightblock'), true );
    }

    protected function prepareDaysFromToday($date, $dateUrl, $urlPostfix = '')
    {
        $aData = array(
            'date'        => $date,
            'date_url'    => $dateUrl,
            'url_postfix' => $urlPostfix,
        );
        return $this->viewPHP($aData, 'common.days.fromtoday');
    }
    
    protected function prepareDaysPeriod($nPeriodStart, $nPeriodFinish, $nOffset, $sLink)
    {
        $aData = array(
            'from'   => $nPeriodStart,
            'to'     => $nPeriodFinish,
            'offset' => $nOffset,
            'link'   => $sLink,
        );
        
        return $this->viewPHP($aData, 'common.days.period');
    }
    
    protected function prepareDate($sDate, &$bToday, $bNowIfEmpty = true)
    {
        $mResult = $bToday = false;
        
        do {
            if ($sDate=='' || strlen($sDate)<10 || !preg_match('/[\d]{4}-[\d]{2}-[\d]{2}/', $sDate)) {
                break;
            }
            $nDate = strtotime( $sDate );
            if ($nDate === -1 || $nDate === false || $nDate < 342133200) //не удалось определить дату, либо дата ранее 1980 года :)
                break;
                
            $mResult = $nDate;
        } while(false);
        
        if ($mResult === false)
        {
            if ($bNowIfEmpty) {
                $mResult = mktime(0,0,0); // текущая дата без времени, Y-m-d 00:00:00
                $bToday = true;
            }
        } else {
            if ( mktime(0,0,0) == strtotime( date('Y-m-d', $nDate) ) ) {
                $bToday = true;
            }
        }
        
        return $mResult;
    }

    protected function prepareMenu($sActiveKey, $bHoverOnly = false) // need texts
    {
        $menu = array(
            'events' => array('t'=>'?', 'a'=>0, 'h'=>static::url('list', $this->type->id)),
            'places' => array('t'=>_t('afisha', 'Места'), 'a'=>0, 'h'=>static::url('places', $this->type->id)),
        );

        switch ($this->type->id)
        {
            case self::TYPE_MOVIE: { // Кино
                $menu['events']['t'] = _t('afisha', 'Фильмы');
                $menu['places']['t'] = _t('afisha', 'Кинотеатры');
                $menu['soon']        = array('t'=>_t('afisha', 'Скоро'), 'a'=>0, 'h'=>static::url('list.tab', array('type'=>$this->type->id, 'tab'=>'soon')));
                $menu['archive']     = array('t'=>_t('afisha', 'Архив'), 'a'=>0, 'h'=>static::url('list.tab', array('type'=>$this->type->id, 'tab'=>'archive')));
            } break;
            case self::TYPE_THEATER: { // Театр
                $menu['events']['t'] = _t('afisha', 'Спектакли');
                $menu['places']['t'] = _t('afisha', 'Театры');
            } break;
            case self::TYPE_EXHIBITION: { // Выставки
                $menu['events']['t'] = _t('afisha', 'Выставки');
                // $menu['places']['t'] = '?';
            } break;
            case self::TYPE_CONCERT: { // Концерты
                $menu['events']['t'] = _t('afisha', 'Концерты');
                $menu['places']['t'] = _t('afisha', 'Площадки');
            } break;
            case self::TYPE_ACTIVITY: { // Мероприятия
                $menu['events']['t'] = _t('afisha', 'Мероприятия');
                // $menu['places']['t'] = '?';
            } break;
            case self::TYPE_SPORT: { // Спорт
                $menu['events']['t'] = _t('afisha', 'Мероприятия');
                // $menu['places']['t'] = '?';
            } break;
            case self::TYPE_PARTY: { // Вечеринки
                $menu['events']['t'] = _t('afisha', 'Вечеринки');
                // $menu['places']['t'] = '?';
            } break;
            case self::TYPE_CIRCUS: { // Цирк
                $menu['events']['t'] = _t('afisha', 'Представления');
                // $menu['places']['t'] = '?';
            } break;
        }

        if (isset($menu[$sActiveKey])) {
            $menu[$sActiveKey]['a'] = ($bHoverOnly ? 2 : 1);
        }

        $aData = array('menu'=>$menu);
        return $this->viewPHP($aData, 'common.menu');
    }

    protected function showVideoBlock($eventID, $videoType, $videoData)
    {
        if ( ! $this->type->videoEnabled()) {
            return '';
        }
        return $this->initVideo( array('id'=>$eventID), $this->type )->show_video_block($videoType, $videoData);
    }
}