<?php namespace bff\utils;

/**
 * Класс вспомогательных методов обработки текста
 * @version 0.36
 * @modified 19.apr.2015
 */

class TextParser
{
    public function __construct()
    {
    }

    /**
     * Инициализация компонента Jevix
     * @return \Jevix
     */
    public function jevix()
    {
        static $i;
        if (!isset($i)) {
            require_once(PATH_CORE . 'external/jevix/jevix.class.php');
            $i = new \Jevix();
        }

        return $i;
    }

    /**
     * Парсинг текста с помощью Jevix
     * @param string $sText текст
     * @param array $aErrors @ref массив ошибок (заполняется Jevix парсером)
     * @return string обработанный текст
     */
    public function JevixParser($sText, &$aErrors = array())
    {
        $sResult = $this->jevix()->parse($sText, $aErrors);

        return $sResult;
    }

    /**
     * Парсинг текста на предмет видео
     * @param string $sText текст
     * @return string
     */
    public function VideoParser($sText)
    {
        # youtube.com
        $sText = preg_replace('/<video>http:\/\/(?:www\.|)youtube\.com\/watch\?v=([a-zA-Z0-9_\-]+)<\/video>/Ui', '<object width="425" height="344"><param name="movie" value="http://www.youtube.com/v/$1&hl=en"></param><param name="wmode" value="opaque"></param><embed src="http://www.youtube.com/v/$1&hl=en" type="application/x-shockwave-flash" wmode="opaque" width="425" height="344"></embed></object>', $sText);
        # rutube.ru
        $sText = preg_replace('/<video>http:\/\/(?:www\.|)rutube.ru\/tracks\/\d+.html\?v=([a-zA-Z0-9_\-]+)<\/video>/Ui', '<object width="470" height="353"><param name="movie" value="http://video.rutube.ru/$1"></param><param name="wmode" value="opaque"></param><param name="allowFullScreen" value="true"></param><param name="flashVars" value="uid=662118"></PARAM><EMBED src="http://video.rutube.ru/$1" type="application/x-shockwave-flash" wmode="opaque" width="470" height="353" allowFullScreen="true" flashVars="uid=662118"></embed></object>', $sText);

        return $sText;
    }

    /**
     * Заменяет все вхождения короткого тега <param/> на длиную версию <param></param>
     * Заменяет все вхождения короткого тега <embed/> на длиную версию <embed></embed>
     * @param string $sText текст
     */
    protected function FlashParamParser($sText)
    {
        if (preg_match_all("@(<\s*param\s*name\s*=\s*\".*\"\s*value\s*=\s*\".*\")\s*/?\s*>(?!</param>)@Ui", $sText, $aMatch)) {
            foreach ($aMatch[1] as $key => $str) {
                $str_new = $str . '></param>';
                $sText = str_replace($aMatch[0][$key], $str_new, $sText);
            }
        }
        if (preg_match_all("@(<\s*embed\s*.*)\s*/?\s*>(?!</embed>)@Ui", $sText, $aMatch)) {
            foreach ($aMatch[1] as $key => $str) {
                $str_new = $str . '></embed>';
                $sText = str_replace($aMatch[0][$key], $str_new, $sText);
            }
        }
        /**
         * Удаляем все <param name="wmode" value="*"></param>
         */
        if (preg_match_all("@(<param\s.*name=\"wmode\".*>\s*</param>)@Ui", $sText, $aMatch)) {
            foreach ($aMatch[1] as $key => $str) {
                $sText = str_replace($aMatch[0][$key], '', $sText);
            }
        }
        /**
         * А теперь после <object> добавляем <param name="wmode" value="opaque"></param>
         * Решение не фантан, но главное работает :)
         */
        if (preg_match_all("@(<object\s.*>)@Ui", $sText, $aMatch)) {
            foreach ($aMatch[1] as $key => $str) {
                $sText = str_replace($aMatch[0][$key], $aMatch[0][$key] . '<param name="wmode" value="opaque"></param>', $sText);
            }
        }

        return $sText;
    }

    /**
     * Делает ссылки не видимыми для поисковиков
     * @param string $sText текст
     * @return string
     */
    public function MakeUrlNoIndex($sText)
    {
        return preg_replace("/(<a .*>.*<\/a>)/Ui", "<noindex>$1</noindex>", $sText);
    }

    /**
     * Парсит текст комментария (без HTML тегов)
     * @param string $sText текст комментария
     * @param integer $nMaxLength максимально допустимое кол-во символов
     * @param boolean $bActivateLinks активировать ссылки
     * @return string
     */
    public function parseCommentPlain($sMessage, $nMaxLength = 0, $bActivateLinks = false)
    {
        $sMessage = preg_replace("/(\<script)(.*?)(script>)/si", '', $sMessage);
        $sMessage = htmlspecialchars($sMessage);
        $sMessage = preg_replace("/(\<)(.*?)(--\>)/mi", '' . nl2br("\\2") . '', $sMessage);
        if (!empty($nMaxLength) && $nMaxLength > 0) {
            $sMessage = mb_substr($sMessage, 0, intval($nMaxLength));
        }
        if ($bActivateLinks) {
            $oParser = new LinksParser();
            $sMessage = $oParser->parse($sMessage);
        }

        return $sMessage;
    }

    /**
     * Парсинг текста комментария (с HTML тегами)
     * @param string $sText текст
     */
    public function parseCommentHTML($sText)
    {
        # Конфигурирует типограф
        $j = $this->jevix();

        # Разрешённые теги
        $j->cfgAllowTags(array(
                'cut',
                'a', 'img',
                'i', 'b', 'u', 's', 'em', 'strong',
                'nobr',
                'ul', 'li', 'ol',
                'sup', 'sub', 'abbr', 'acronym',
                'h4', 'h5', 'h6',
                'br', 'hr',
                'pre', 'code',
                'object', 'param', 'embed', 'video',
                'blockquote',
            )
        );
        # Коротие теги типа
        $j->cfgSetTagShort(array('br', 'img', 'hr', 'cut'));
        # Преформатированные теги
        $j->cfgSetTagPreformatted(array('pre', 'code', 'video'));
        # Разрешённые параметры тегов
        $j->cfgAllowTagParams('img', array(
                'src',
                'alt'    => '#text',
                'title',
                'align'  => array('right', 'left', 'center'),
                'width'  => '#int',
                'height' => '#int',
                'hspace' => '#int',
                'vspace' => '#int'
            )
        );
        $j->cfgAllowTagParams('a', array('href', 'title', 'rel'));
        $j->cfgAllowTagParams('cut', array('name'));
        $j->cfgAllowTagParams('object', array('width' => '#int', 'height' => '#int', 'data' => '#link'));
        $j->cfgAllowTagParams('param', array('name' => '#text', 'value' => '#text'));
        $j->cfgAllowTagParams('embed', array(
                'src'               => '#image',
                'type'              => '#text',
                'allowscriptaccess' => '#text',
                'allowfullscreen'   => '#text',
                'width'             => '#int',
                'height'            => '#int',
                'flashvars'         => '#text',
                'wmode'             => '#text'
            )
        );
        # Параметры тегов являющиеся обязательными
        $j->cfgSetTagParamsRequired('img', 'src');
        $j->cfgSetTagParamsRequired('a', 'href');
        # Теги которые необходимо вырезать из текста вместе с контентом
        $j->cfgSetTagCutWithContent(array('script', 'iframe', 'style'));
        # Вложенные теги
        $j->cfgSetTagChilds('ul', array('li'), false, true);
        $j->cfgSetTagChilds('ol', array('li'), false, true);
        $j->cfgSetTagChilds('object', 'param', false, true);
        $j->cfgSetTagChilds('object', 'embed', false, false);
        # Если нужно оставлять пустые не короткие теги
        $j->cfgSetTagIsEmpty(array('param', 'embed'));
        # Теги с обязательными параметрами
        $j->cfgSetTagParamsAutoAdd('embed', array(array('name' => 'wmode', 'value' => 'opaque', 'rewrite' => true)));
        # Отключение авто-добавления <br>
        $j->cfgSetAutoBrMode(false);
        # Автозамена
        $j->cfgSetAutoReplace(array('+/-', '(c)', '(r)', '(C)', '(R)'), array('±', '©', '®', '©', '®'));
        $j->cfgSetXHTMLMode(false);
        $j->cfgSetTagNoTypography('code');
        $j->cfgSetTagNoTypography('video');

        $sResult = $this->FlashParamParser($sText);
        $sResult = $this->JevixParser($sResult);
        $sResult = $this->VideoParser($sResult);

        return $sResult;
    }

    /**
     * Парсинг wysiwyg текста
     * Метод используется компонентом {bff\db\Publicator}
     * @param string $sText текст
     * @param array $aParams доп. настройки:
     *   boolean 'scripts' - разрешать вставку script тегов
     *   boolean 'iframes' - разрешать вставку iframe тегов
     *   array 'links_parser' - настройки обработки ссылок
     * @return string
     */
    public function parseWysiwygText($sText, $aParams = array())
    {
        static $configured;

        $j = $this->jevix();

        if (!isset($configured)) {
            $configured = true;

            # 1. Разрешённые теги. (Все не разрешенные теги считаются запрещенными.)
            $allowedTags = array(
                'a', 'img',
                'i', 'b', 'u', 's', 'em', 'strong', 'small', 'font',
                'nobr', 'map', 'area', 'col', 'colgroup',
                'ul', 'li', 'ol',
                'dd', 'dl', 'dt',
                'sub', 'sup', 'abbr', 'acronym',
                'pre', 'code',
                'h1', 'h2', 'h3', 'h4', 'h5', 'h6',
                'div', 'p', 'span', 'br', 'hr',
                'object', 'param', 'embed', 'video', 'audio', 'source', 'track',
                'blockquote', 'q', 'caption',
                'table', 'thead', 'tbody', 'tfoot', 'tr', 'th', 'td',
                # form
                'form', 'input', 'button', 'textarea', 'noscript', 'select', 'opt', 'option', 'optgroup',
                'fieldset', 'label', 'legend',
                # html5:
                'article', 'aside', 'bdi', 'bdo', 'details', 'dialog', 'figcaption', 'figure',
                'footer', 'header', 'main', 'mark', 'menu', 'menuitem', 'meter', 'nav', 'progress',
                'rp', 'rt', 'ruby', 'section', 'summary', 'time', 'wbr',
                'datalist', 'keygen', 'output', 'canvas', 'svg',
            );
            $j->cfgAllowTags($allowedTags);

            # 2. Коротие теги. (не имеющие закрывающего тега)
            $j->cfgSetTagShort(array('br', 'img', 'hr'));

            # 3. Преформатированные теги. (в них все будет заменятся на HTML сущности)
            $j->cfgSetTagPreformatted(array('pre'));

            # 4. Теги, которые необходимо вырезать из текста вместе с контентом.
            if (!empty($aParams['scripts'])) {
                $j->cfgAllowTags(array('script')); $allowedTags[] = 'script';
                $j->cfgSetTagIsEmpty(array('script','div','span'));
                $j->cfgAllowTagParams('script', array('src', 'type', 'charset', 'async', 'defer'));
                $j->cfgSetTagCallback('script', function($content){return $content;});
            } else {
                $j->cfgSetTagCutWithContent(array('script'));
            }
            if (!empty($aParams['iframes'])) {
                $j->cfgAllowTags(array('iframe')); $allowedTags[] = 'iframe';
                $j->cfgSetTagIsEmpty(array('iframe'));
                $j->cfgAllowTagParams('iframe', array(
                        'name', 'align', 'src', 'frameborder',
                        'height', 'width', 'scrolling',
                        'marginwidth', 'marginheight'
                    )
                );
            } else {
                $j->cfgSetTagCutWithContent(array('iframe'));
            }
            $j->cfgSetTagCutWithContent(array('style'));

            # 5. Разрешённые параметры тегов. Также можно устанавливать допустимые значения этих параметров.
            $j->cfgAllowTagParams('a', array('title', 'href', 'target', 'rel'));
            $j->cfgAllowTagParams('img', array(
                    'src',
                    'alt'    => '#text',
                    'title',
                    'align'  => array('right', 'left', 'center'),
                    'width'  => '#int',
                    'height' => '#int'
                )
            );

            # specials:
            $j->cfgAllowTagParams('blockquote', array('data-instgrm-captioned', 'data-instgrm-version'));
            $j->cfgAllowTagParams('font', array('color'));

            # allow: style, class, id, lang
            foreach ($allowedTags as $tag) {
                $j->cfgAllowTagParams($tag, array('style', 'class', 'id', 'lang'));
            }

            # allow: align
            foreach (array('span','div','p','blockquote') as $tag) {
                $j->cfgAllowTagParams($tag, array('align'));
            }

            # 6. Параметры тегов являющиеся обязяательными. Без них вырезает тег оставляя содержимое.
            $j->cfgSetTagParamsRequired('img', 'src');

            # 7. Теги которые может содержать тег контейнер
            //    cfgSetTagChilds($tag, $childs, $isContainerOnly, $isChildOnly)
            //       $isContainerOnly : тег является только контейнером для других тегов и не может содержать текст (по умолчанию false)
            //       $isChildOnly : вложенные теги не могут присутствовать нигде кроме указанного тега (по умолчанию false)
            $j->cfgSetTagChilds('ul', 'li', true, false);
            $j->cfgSetTagChilds('ol', 'li', true, false);

            # 8. Атрибуты тегов, которые будут добавлятся автоматически
            $j->cfgSetLinkProtocolAllow(array('mailto','skype'));
            //$j->cfgSetTagParamsAutoAdd('a', array('rel' => 'nofollow'));
            //$j->cfgSetTagParamsAutoAdd('a', array('name'=>'rel', 'value' => 'nofollow', 'rewrite' => true));

            $j->cfgSetTagParamDefault('img', 'width', '565px');

            # 9. Автозамена
            $j->cfgSetAutoReplace(array('+/-', '(c)', '(r)'), array('±', '©', '®'));

            # 10. Включаем режим XHTML. (по умолчанию включен)
            $j->cfgSetXHTMLMode(true);

            # 11. Выключаем режим замены переноса строк на тег <br/>. (по умолчанию включен)
            $j->cfgSetAutoBrMode(false);

            # 12. Включаем режим автоматического определения ссылок. (по умолчанию включен)
            $j->cfgSetAutoLinkMode(true);

            # 13. Отключаем типографирование в определенном теге
            $j->cfgSetTagNoTypography(array('code','video'));
        }

        $sText = str_replace('&nbsp;', ' ', $sText);

        # Подсвечиваем внешние ссылки
        if (!empty($aParams['links_parser']) && is_array($aParams['links_parser'])) {
            if (isset($aParams['links_parser']['highlight-new']) && !$aParams['links_parser']['highlight-new']) {
                $j->cfgSetAutoLinkMode(false);
            }
            $sText = $this->JevixParser($sText, $aErrors);
            $linksParser = new \bff\utils\LinksParser();
            return $linksParser->parse($sText, $aParams['links_parser']);
        } else {
            return $this->JevixParser($sText, $aErrors);
        }
    }

    /**
     * Парсинг wysiwyg текста при публикации с фроненда
     * @param string $sText текст
     * @param array $aParams доп. настройки:
     *  string 'img-default-width' - ширина изображения по-умолчанию (если не указана)
     * @return string
     */
    public function parseWysiwygTextFrontend($sText, $aParams = array())
    {
        static $configured;

        if (!isset($configured)) {
            $configured = true;
            $j = $this->jevix();

            # 1. Разрешённые теги. (Все неразрешенные теги считаются запрещенными.)
            $j->cfgAllowTags(array(
                    'a',
                    'img',
                    'i',
                    'b',
                    'u',
                    'em',
                    'strong',
                    'nobr',
                    'li',
                    'ol',
                    'ul',
                    'sub',
                    'sup',
                    'abbr',
                    'pre',
                    'acronym',
                    'h1', 'h2', 'h3', 'h4', 'h5', 'h6',
                    'br',
                    'hr',
                    'p',
                    'span',
                    'div',
                    'code',
                    'blockquote',
                    'table', 'thead', 'tbody', 'tfoot', 'tr', 'th', 'td',
                )
            );

            # 2. Коротие теги. (не имеющие закрывающего тега)
            $j->cfgSetTagShort(array('br', 'img', 'hr'));

            # 3. Преформатированные теги. (в них все будет заменятся на HTML сущности)
            $j->cfgSetTagPreformatted(array('pre'));

            # 4. Теги, которые необходимо вырезать из текста вместе с контентом.
            $j->cfgSetTagCutWithContent(array('style','script','iframe'));

            # 5. Разрешённые параметры тегов. Также можно устанавливать допустимые значения этих параметров.
            $j->cfgAllowTagParams('a', array('title', 'href', 'target', 'rel'));
            $j->cfgAllowTagParams('img', array(
                    'class',
                    'src',
                    'alt'    => '#text',
                    'title',
                    'align'  => array('right', 'left', 'center'),
                    'width'  => '#int',
                    'height' => '#int'
                )
            );
            $j->cfgAllowTagParams('span', array('align', 'class'));
            $j->cfgAllowTagParams('div', array('align', 'class'));
            $j->cfgAllowTagParams('ul', array('class'));
            $j->cfgAllowTagParams('li', array('class'));
            $j->cfgAllowTagParams('table', array('style', 'class'));
            $j->cfgAllowTagParams('tr', array('class'));
            $j->cfgAllowTagParams('th', array('class'));
            $j->cfgAllowTagParams('td', array('class'));

            # 6. Параметры тегов являющиеся обязяательными. Без них вырезает тег оставляя содержимое.
            $j->cfgSetTagParamsRequired('img', 'src');

            # 7. Теги которые может содержать тег контейнер
            $j->cfgSetTagChilds('ul', 'li', true, false);
            $j->cfgSetTagChilds('ol', 'li', true, false);

            # 8. Атрибуты тегов, которые будут добавлятся автоматически
            $j->cfgSetTagParamDefault('a', 'rel', null, true);
            $j->cfgSetLinkProtocolAllow(array('mailto','skype'));
            if (!empty($aParams['img-default-width'])) {
                $j->cfgSetTagParamDefault('img', 'width', $aParams['img-default-width']);
            }

            # 9. Автозамена
            $j->cfgSetAutoReplace(array('+/-', '(c)', '(r)'), array('±', '©', '®'));

            # 10. Включаем режим XHTML.
            $j->cfgSetXHTMLMode(true);

            # 11. Выключаем режим замены переноса строк на тег <br/>.
            $j->cfgSetAutoBrMode(false);

            # 12. Включаем режим автоматического определения ссылок.
            $j->cfgSetAutoLinkMode(false);

            # 13. Отключаем типографирование в определенном теге
            $j->cfgSetTagNoTypography('code');
        }

        $sText = nl2br(preg_replace("/\>(\r\n|\r|\n)/u", '>', $sText));
        $sText = str_replace('&nbsp;', ' ', $sText);

        return $this->JevixParser($sText, $aErrors);
    }

    /**
     * Простой метод корректировки неправильной раскладки клавиатуры
     * @param string $string строка, требующая корректировки
     * @param string $from раскладка в которой предположительно набирался текст
     * @param string $to раскладка в которую необходимо конвертировать
     * @return string
     */
    public function correctKeyboardLayout($string, $from = 'en', $to = 'ru')
    {
        static $data = array(
            'en' => array(
                'q',
                'w',
                'e',
                'r',
                't',
                'y',
                'u',
                'i',
                'o',
                'p',
                '[',
                ']',
                "\\",
                'a',
                's',
                'd',
                'f',
                'g',
                'h',
                'j',
                'k',
                'l',
                ';',
                "'",
                'z',
                'x',
                'c',
                'v',
                'b',
                'n',
                'm',
                ',',
                '.'
            ),
            'ru' => array(
                'й',
                'ц',
                'у',
                'к',
                'е',
                'н',
                'г',
                'ш',
                'щ',
                'з',
                'х',
                'ъ',
                'ё',
                'ф',
                'ы',
                'в',
                'а',
                'п',
                'р',
                'о',
                'л',
                'д',
                'ж',
                'э',
                'я',
                'ч',
                'с',
                'м',
                'и',
                'т',
                'ь',
                'б',
                'ю'
            ),
            'ua' => array(
                'й',
                'ц',
                'у',
                'к',
                'е',
                'н',
                'г',
                'ш',
                'щ',
                'з',
                'х',
                'ї',
                'ґ',
                'ф',
                'и',
                'в',
                'а',
                'п',
                'р',
                'о',
                'л',
                'д',
                'ж',
                'є',
                'я',
                'ч',
                'с',
                'м',
                'і',
                'т',
                'ь',
                'б',
                'ю'
            ),
        );
        if (!isset($data[$from]) || !isset($data[$to])) {
            return $string;
        }

        return preg_replace($data[$from], $data[$to], mb_strtolower($string));
    }

}