<?php

namespace package;

use core\Request;
use core\View;

class Module extends \Package
{
    protected function onCheckComponent($component)
    {

    }
    protected function onUpdateComponent($component)
    {

    }
    protected function onLoadComponent(\Component $component)
    {
        if ($component->isEnabled()) {
            $path_file_module = $component->getName() . '/' . $component->getName() . '.module.php';
            if(is_file($this->__path_dir . '/' . $path_file_module)) {
                require_once $this->__path_dir . '/' . $path_file_module;
            }
        }
    }

    protected function onUpdate()
    {

    }
    protected function onLoad()
    {

    }

    protected function getType()
    {
        return 'module';
    }
    protected function getClass()
    {
        return \component\Module::class;
    }

    const PATH_VARIABLE_PATTERN = '#\{([^\/]+?)\}#';

    /**
     * Module constructor.
     * @param $path_dir
     */
    public function __construct($path_dir)
    {
        $this->__types = array_merge($this->__types, ['routes']);
        parent::__construct($path_dir);
    }

    /**
     * @param Request $request
     * @return array|mixed
     * @throws \Exception
     */
    public function execute(Request $request) {

        foreach ($this->getComponents() AS $component_name => $component) {
            if (!$component->isEnabled()) {
                continue;
            }

            if (!$routes = $component->findConfig('routes')) {
                continue;
            }

            foreach ($routes AS $route_name => $route) {
                if(isset($route['methods'])) {
                    if (!$request->isMethod($route['methods'])) {
                        continue;
                    }
                }

                if (isset($route['requirements'])) {
                    $requirements = $route['requirements'];
                    if (isset($requirements['is_logged'])) {
                        if(\User::isAuth() != $requirements['is_logged']) {
                            continue;
                        }
                    }

                    if (isset($requirements['is_admin'])) {
                        if(\User::isAuth() != true || \User::getAccount()->is_admin != $requirements['is_admin']) {
                            continue;
                        }
                    }
                }

                if(!isset($route['path']) || !isset($route['config'])) {
                    continue;
                }

                $path = $route['path'];
                $config = $route['config'];

                if ($this->hasExecuteRequest($path, $request)) {
                    if(isset($config['controller'])) {
                        $path_controller = '/' . $component_name . '/controllers/' . $config['controller'] . '.php';
                        if (!is_file($this->__path_dir . $path_controller)) {
                            continue;
                        }

                        $view = new View($this->__path_dir . '/' . $component->getName() . '/views');
                        extract([$component, $request, $view, $this]);

                        ob_start();
                        $result = include $this->__path_dir . $path_controller;
                        $content = ob_get_clean();

                        if($content) {
                            $result = $content;
                        }

                        return [
                            'title' => $config['title'] ?? null,
                            'content' => $result
                        ];
                    }
                }
            }
        }

        return false;
    }

    /**
     * @param string $pattern
     * @param Request $request
     * @return bool
     */
    protected function hasExecuteRequest(string $pattern, Request $request)
    {
        return !!preg_match($this->getPregPatternUrl($pattern), $request->getUrl()->getPath());
    }

    /**
     * @param string $pattern
     * @return string
     */
    protected function getPregPatternUrl(string $pattern)
    {
        // статичные участки пути
        $static_paths = preg_split(self::PATH_VARIABLE_PATTERN, $pattern);

        // переменные участки пути, расположенные между статичных
        $vars_matches = $this->getPathVariablesMatches($pattern);

        // собираем регулярку для обработки пути
        $pattern = '';
        for ($i = 0; $i < count($static_paths); $i++) {
            $pattern .= preg_quote($static_paths[$i], '#'); // статичные участки экранируем

            if (array_key_exists($i, $vars_matches)) {
                // переменных частей должно быть на одну меньше, чем статичных? поэтому на всякий случай проверяем
                $pattern .= $vars_matches[$i];
            }
        }

        return '#^' . $pattern . '$#';
    }

    /**
     * @param string $pattern
     * @return array
     */
    protected function getPathVariablesMatches(string $pattern)
    {
        $matches = [];
        $vars = $this->getPathVariables($pattern);
        foreach ($vars AS $var_name) {
            switch ($var_name) {
                case '**':
                    $matches[] = '(.+)';
                    break;
                case '*':
                default:
                    $matches[] = '([^/]+)';
                    break;
            }
        }
        return $matches;
    }

    /**
     * @param string $pattern
     * @return mixed
     */
    protected function getPathVariables(string $pattern)
    {
        $vars = [];
        preg_match_all(self::PATH_VARIABLE_PATTERN, $pattern, $vars, PREG_PATTERN_ORDER);
        return $vars[1];
    }
}