<?php

abstract class Package
{
    protected static $__connection_update;
    protected static $__connection_load;
    protected static $__connection_installed;
    protected static $__connection_enabled;

    protected $__components = [];
    protected $__path_dir;
    protected $__types = ['info'];

    protected abstract function getType();
    protected abstract function getClass();

    protected final function __update()
    {
        if ($dh = opendir($this->__path_dir)) {
            while (($name = readdir($dh)) !== false) {
                if ($name == '.' || $name == '..') {
                    continue;
                }

                $component = [];
                $component['name'] = $name;

                foreach ($this->__types AS $type) {
                    $path_file = $this->__path_dir . '/' . $name . '/' . $name . '.' . $type . '.json';
                    if (is_file($path_file)) {
                        $file_json = file_get_contents($path_file);
                        $component[$type] = \Json::parse($file_json);
                    }
                }

                $this->__checkComponent($component);
                $this->__updateComponent($component);
            }

            closedir($dh);
            $this->onUpdate();
        }
    }
    protected abstract function onUpdate();

    protected final function __load()
    {
        if (!$this->__components) {
            self::$__connection_load->execute([$this->getType()]);
            $components = self::$__connection_load->fetchAll(\PDO::FETCH_UNIQUE|\PDO::FETCH_CLASS,$this->getClass());
            $this->__components = $components;

            array_map([$this, '__loadComponent'], $components);

            $this->onLoad();
        }
    }
    protected abstract function onLoad();

    protected final function __checkComponent($component)
    {
        if (!array_key_exists('type', $component['info'])) {
            throw new \Exception(sprintf("Не указан тип пакета\n%s",
                \Json::stringify($component['info'])));
        }

        if ($component['info']['type'] !== $this->getType()) {
            throw new \Exception(sprintf("Тип пакета должен быть %s (указан %s)\n%s",
                $this->getType(),
                $component['info']['type'],
                \Json::stringify($component['info'])));
        }

        if (!array_key_exists('title', $component['info'])) {
            throw new \Exception(sprintf("Не указано имя пакета\n%s",
                \Json::stringify($component['info'])));
        }

        if (!array_key_exists('version', $component['info'])) {
            throw new \Exception(sprintf("Не указана версия пакета\n%s",
                \Json::stringify($component['info'])));
        }

        $this->onCheckComponent($component);
    }
    protected abstract function onCheckComponent($component);

    protected final function __updateComponent($component)
    {
        if (self::$__connection_update) {
            $system = $component['info']['system'] ?? 0;
            self::$__connection_update->execute([
                ':type' => $this->getType(),
                ':name' => $component['name'],
                ':version' => $component['info']['version'],
                ':installed' => $system,
                ':enabled' => $system,
                ':system' => $system,
                ':config' => \Json::stringify($component, false)
            ]);
        }

        $this->onUpdateComponent($component);
    }
    protected abstract function onUpdateComponent($component);

    protected final function __loadComponent(Component $component)
    {
        $this->onLoadComponent($component);
    }
    protected abstract function onLoadComponent(Component $component);

    /**
     * Package constructor.
     * @param $path_dir
     */
    public function __construct($path_dir)
    {
        $connection = Registry::get('connection');
        if ($connection || !self::$__connection_update) {
            /** @var \PDO $connection */
            self::$__connection_update = $connection->prepare('INSERT INTO `components` (
            `type`, 
            `name`, 
            `version`, 
            `installed`, 
            `enabled`, 
            `system`, 
            `config`)
          VALUES (
            :type, 
            :name,
            :version, 
            :installed,
            :enabled, 
            :system, 
            :config)
          ON DUPLICATE KEY UPDATE 
            `version` = VALUES(`version`), 
            `system` = VALUES(`system`), 
            `config` = VALUES(`config`)');
        }
        if ($connection || !self::$__connection_load) {
            /** @var \PDO $connection */
            self::$__connection_load = $connection->prepare("SELECT 
                  name, 
                  name, 
                  type, 
                  version, 
                  config, 
                  system,
                  enabled,
                  installed
                FROM `components` WHERE `type` = ?");
        }

        $this->__path_dir = $path_dir;
        $this->__update();
        $this->__load();
    }

    /**
     * @return Component[]
     */
    public function getComponents()
    {
        return $this->__components;
    }

    /**
     * @param string $name
     * @return Component
     * @throws Exception
     */
    public function findComponent(string $name)
    {
        if (!$this->hasComponent($name)) {
            throw new \Exception(sprintf("Пакет %s не найден", $name));
        }

        $components = $this->getComponents();
        return $components[$name];
    }

    /**
     * Проверяет на наличие пакета
     * @param $name
     * @return bool
     */
    public function hasComponent($name)
    {
        return key_exists($name, $this->getComponents());
    }

    /**
     * Возвращает версию пакета
     * @param $name
     * @return string
     * @throws \Exception
     */
    public function getVersion($name)
    {
        $component = $this->findComponent($name);
        return $component->getVersion();
    }

    /**
     * Флаг, установлен ли пакет
     * @param $name
     * @return bool
     * @throws \Exception
     */
    public function getIsInstalled($name)
    {
        $component = $this->findComponent($name);
        return $component->isInstalled();
    }

    /**
     * Флаг, включен ли пакет
     * @param $name
     * @return bool
     * @throws \Exception
     */
    public function getIsEnabled($name)
    {
        $component = $this->findComponent($name);
        return $component->isEnabled();
    }
}