<?php

namespace core;

class Request
{

    protected $url;

    /**
     * Request constructor.
     * @param string|null $url
     */
    public function __construct($url = null)
    {
        if(is_string($url)) {
            $url = new Url($url);
        }

        if (is_null($url)) {
            $url = new Url();
        }

        $this->url = $url;
    }

    /**
     * @return Url
     */
    public function getUrl()
    {
        return $this->url;
    }

    /**
     * @param string $name
     * @param int $filter
     * @return mixed
     */
    public function post(string $name, $filter = FILTER_DEFAULT)
    {
        return filter_input(INPUT_POST, $name, $filter);
    }

    /**
     * @param string $name
     * @param int $filter
     * @return mixed
     */
    public function get(string $name, $filter = FILTER_DEFAULT)
    {
        return filter_input(INPUT_GET, $name, $filter);
    }

    /**
     * @return string
     */
    public function getMethod()
    {
        return $_SERVER['REQUEST_METHOD'];
    }

    /**
     * @return bool
     */
    public function isMethodPost(): bool
    {
        return $this->getMethod() == 'POST';
    }

    /**
     * @param string $method
     * @return bool
     */
    public function isValidMethod(string $method): bool
    {
        $method = strtoupper($method);
        switch ($method) {
            case "GET":
            case "POST":
            case "PUT":
            case "DELETE":
            case "HEAD":
            case "OPTIONS":
            case "PATCH":
                return true;
        }

        return false;
    }

    /**
     * @param array $methods
     * @return bool
     * @throws \Exception
     */
    public function isMethod(array $methods): bool
    {
        foreach ($methods AS $method) {
            if (!$this->isValidMethod($method)) {
                if (is_string($method)) {
                    throw new \Exception("Invalid HTTP method: " . $method);
                } else {
                    throw new \Exception("Invalid HTTP method: non-string");
                }
            }

            if ($method == $this->getMethod()) {
                return true;
            }
        }

        return false;
    }
}