<?php

namespace core\account;

use data\Account_Achievement;
use data\Account_Profession;
use data\Account_Quest;

class Event
{
    protected $account_id = 0;

    protected $quests = [];
    protected $achievements = [];
    protected $professions = [];

    /**
     * Craft constructor.
     * @param int $account_id
     */
    public function __construct($account_id = 0)
    {
        $this->account_id = $account_id;
    }

    /**
     * @return Account_Quest[]
     */
    public function getQuests()
    {
        if (!$this->quests) {
            /** @var \PDO $connection */
            $connection = \Registry::get('connection');

            $res = $connection->prepare("SELECT quest_id, id 
            FROM `account_quests` 
            WHERE `account_id` = ?");
            $res->execute([$this->account_id]);
            $this->quests = $res->fetchAll(\PDO::FETCH_UNIQUE | \PDO::FETCH_CLASS, Account_Quest::class);
        }
        return $this->quests;
    }

    /**
     * @return Account_Achievement[]
     */
    public function getAchievements()
    {
        if (!$this->achievements) {
            /** @var \PDO $connection */
            $connection = \Registry::get('connection');
            $res = $connection->prepare("SELECT achievement_id, id, step, achievement_category
            FROM `account_achievements` 
            WHERE `account_id` = ?");
            $res->execute([$this->account_id]);
            $this->achievements = $res->fetchAll(\PDO::FETCH_UNIQUE | \PDO::FETCH_CLASS, Account_Achievement::class);
        }
        return $this->achievements;
    }

    /**
     * @return Account_Profession[]
     */
    public function getProfessions()
    {
        if (!$this->professions) {
            /** @var \PDO $connection */
            $connection = \Registry::get('connection');

            $res = $connection->prepare("SELECT profession_id, id 
            FROM `account_professions`
            WHERE `account_id` = ?");
            $res->execute([$this->account_id]);
            $this->professions = $res->fetchAll(\PDO::FETCH_UNIQUE | \PDO::FETCH_CLASS, Account_Profession::class);
        }
        return $this->professions;
    }

    /**
     * @param int $quest_id
     * @return Account_Quest
     */
    public function findQuest(int $quest_id)
    {
        if (!isset($this->quests[$quest_id])) {
            /** @var \PDO $connection */
            $connection = \Registry::get('connection');

            $res = $connection->prepare("SELECT * 
            FROM `account_quests` 
            WHERE `quest_id` = ? AND `account_id` = ? LIMIT 1");
            $res->execute([$quest_id, $this->account_id]);
            $this->quests[$quest_id] = $res->fetchObject(Account_Quest::class);
        }
        return $this->quests[$quest_id];
    }

    /**
     * @param int $achievement_id
     * @return Account_Achievement
     */
    public function findAchievement(int $achievement_id)
    {
        if (!isset($this->achievements[$achievement_id])) {
            /** @var \PDO $connection */
            $connection = \Registry::get('connection');

            $res = $connection->prepare("SELECT * 
            FROM `account_achievements` 
            WHERE `achievement_id` = ? AND `account_id` = ? LIMIT 1");
            $res->execute([$achievement_id, $this->account_id]);
            $this->achievements[$achievement_id] = $res->fetchObject(Account_Achievement::class);
        }
        return $this->achievements[$achievement_id];
    }

    /**
     * @param int $profession_id
     * @return Account_Profession
     */
    public function findProfession(int $profession_id)
    {
        if (!isset($this->professions[$profession_id])) {
            /** @var \PDO $connection */
            $connection = \Registry::get('connection');

            $res = $connection->prepare("SELECT * FROM `account_professions` 
            WHERE `profession_id` = ? AND `account_id` = ? LIMIT 1");
            $res->execute([$profession_id, $this->account_id]);
            $this->professions[$profession_id] = $res->fetchObject(Account_Profession::class);
        }
        return $this->professions[$profession_id];
    }

    /**
     * @param int $quest_id
     * @return Account_Quest
     */
    public function addQuest(int $quest_id)
    {
        return new Account_Quest($quest_id, $this->account_id);
    }

    /**
     * @param int $achievement_id
     * @param int $achievement_category
     * @return Account_Achievement
     */
    public function addAchievement(int $achievement_id, int $achievement_category)
    {
        if(!$this->achievements[$achievement_id]) {
            return $this->achievements[$achievement_id] = new Account_Achievement($achievement_id, $achievement_category, $this->account_id);
        }
        return $this->achievements[$achievement_id];
    }

    /**
     * @param int $profession_id
     * @return Account_Profession
     */
    public function addProfession(int $profession_id)
    {
        return new Account_Profession($profession_id, $this->account_id);
    }
}