<?php

namespace data\account;

class Inventory
{
    protected $id = 0;
    protected $items = [];
    protected $all_items = [];
    protected $slot = 0;

    /**
     * Inventory constructor.
     * @param int $id
     * @param int $slot
     */
    public function __construct($id = 0, $slot = 0)
    {
        $this->id = $id;
        $this->slot = $slot;
    }

    /**
     * @param bool $is_cube
     * @return \array[]
     */
    public function getItems(bool $is_cube = false)
    {
        if (!$this->items) {
            $res = \Core::getConnection()->prepare("SELECT id, id, count, type, account_id, item_id 
            FROM `inventory_items` 
            WHERE `account_id` = ? AND `is_cube` = ?");
            $res->execute([$this->id, $is_cube]);
            $this->items = $res->fetchAll(\PDO::FETCH_UNIQUE|\PDO::FETCH_ASSOC);
        }
        return $this->items;
    }

    /**
     * @return array
     */
    public function getAllItems()
    {
        if (!$this->all_items) {
            $res = \Core::getConnection()->prepare("SELECT id, id, count, type, account_id, item_id, is_cube
            FROM `inventory_items` 
            WHERE `account_id` = ?");
            $res->execute([$this->id]);
            $this->all_items = $res->fetchAll(\PDO::FETCH_UNIQUE|\PDO::FETCH_ASSOC);
        }
        return $this->all_items;
    }

    /**
     * @param int $id
     * @return array
     */
    public function findItem(int $id = 0) {
        if(!isset($this->items[$id])) {
            $res = \Core::getConnection()->prepare("SELECT * FROM `inventory_items` 
            WHERE `id` = ? AND `account_id` = ? LIMIT 1");
            $res->execute([$id, $this->id]);
            $this->items[$id] = $res->fetch(\PDO::FETCH_ASSOC);
        }
        return $this->items[$id];
    }

    /**
     * @param array ...$add_items
     * @return bool
     */
    public function addItem(...$add_items)
    {
        $inventory_items = $this->getAllItems();
        $inventory_updates = [];
        $inventory_count = count($inventory_items);

        foreach ($add_items AS &$item) {
            $gift = $item['gift'] ?? false;
            foreach ($inventory_items AS $inventory_item) {
                if ($gift != $inventory_item['type'] || $item['item_id'] != $inventory_item['item_id']) {
                    continue;
                }
                $data_item = \Core::findItem($inventory_item['item_id']);
                $count = $data_item->count - $inventory_item['count'];
                if (!$count) {
                    continue;
                }
                $item_count = isset($item['count']) ? min($data_item->count, max(1, $item['count'])) : 1;
                if ($count < $item_count) {
                    $inventory_updates[] = [
                        'id' => $inventory_item['id'],
                        'item_id' => $inventory_item['item_id'],
                        'count' => $data_item->count
                    ];
                    $item['count'] = $item_count - $count;
                } else {
                    $inventory_updates[] = [
                        'id' => $inventory_item['id'],
                        'item_id' => $inventory_item['item_id'],
                        'count' => $inventory_item['count'] + $item_count
                    ];
                    unset($item);
                    break;
                }
            }
            if (isset($item)) {
                if(!isset($item['is_cube'])) {
                    $inventory_count += 1;
                }
                $inventory_updates[] = [
                    'item_id' => $item['item_id'],
                    'count' => $item['count'],
                    'gift' => $gift,
                    'is_cube' => $item['is_cube'] ?? false
                ];
            }
        }

        if ($inventory_count > $this->slot) {
            return false;
        }

        foreach ($inventory_updates AS $inventory_update) {
            if (isset($inventory_update['id'])) {
                $res = \Core::getConnection()->prepare("UPDATE `inventory_items` 
                SET `count` = ? 
                WHERE `account_id` = ? AND`id` = ? LIMIT 1");
                $res->execute([
                    $inventory_update['count'],
                    $this->id,
                    $inventory_update['id']
                ]);
            } else {
                $res = \Core::getConnection()->prepare("INSERT INTO `inventory_items` 
                (`account_id`, `item_id`, `count`, `type`, `is_cube`) VALUES (?, ?, ?, ?, ?)");
                $res->execute([
                    $this->id,
                    $inventory_update['item_id'],
                    $inventory_update['count'],
                    ($inventory_update['gift'] ? 1 : 0),
                    ($inventory_update['is_cube'] ? 1 : 0)
                ]);
            }
        }

        return true;
    }

    /**
     * @param array ...$delete_items
     * @return bool
     */
    public function deleteItem(...$delete_items)
    {
        $inventory_items = array_reverse($this->getAllItems());
        $inventory_updates = [];

        foreach ($delete_items AS &$item) {
            foreach ($inventory_items AS $inventory_item) {
                if ($item['item_id'] == $inventory_item['item_id']) {
                    if ($inventory_item['count'] < $item['count']) {
                        $inventory_updates[] = [
                            'id' => $inventory_item['id'],
                            'item_id' => $inventory_item['item_id'],
                            'count' => 0
                        ];

                        $item['count'] = $item['count'] - $inventory_item['count'];
                    } else {
                        $inventory_updates[] = [
                            'id' => $inventory_item['id'],
                            'item_id' => $inventory_item['item_id'],
                            'count' => $inventory_item['count'] - $item['count']
                        ];
                        unset($item);
                        break;
                    }
                }
            }

            if (isset($item)) {
                return false;
            }
        }

        foreach ($inventory_updates AS $inventory_update) {
            if ($inventory_update['count']) {
                $res = \Core::getConnection()->prepare("UPDATE `inventory_items` SET `count` = ? WHERE `account_id` = ? AND`id` = ? LIMIT 1");
                $res->execute([$inventory_update['count'], $this->id, $inventory_update['id']]);
            } else {
                $res = \Core::getConnection()->prepare("DELETE FROM `inventory_items` WHERE `account_id` = ? AND`id` = ?");
                $res->execute([$this->id, $inventory_update['id']]);
            }
        }

        return true;
    }
}