<?php

namespace data\account;

use data\account\event\Achievement;
use data\account\event\Profession;
use data\account\event\Quest;

class Event
{
    protected $id = 0;

    protected $quests = [];
    protected $achievements = [];
    protected $professions = [];

    /**
     * Craft constructor.
     * @param int $id
     */
    public function __construct($id = 0)
    {
        $this->id = $id;
    }

    /**
     * @return Quest[]
     */
    public function getQuests()
    {
        if (!$this->quests) {
            $res = \Core::getConnection()->prepare("SELECT quest_id, id 
            FROM `event_quests` 
            WHERE `account_id` = ?");
            $res->execute([$this->id]);
            $this->quests = $res->fetchAll(\PDO::FETCH_UNIQUE | \PDO::FETCH_CLASS, Quest::class);
        }
        return $this->quests;
    }

    /**
     * @return Achievement[]
     */
    public function getAchievements()
    {
        if (!$this->achievements) {
            $res = \Core::getConnection()->prepare("SELECT achievement_id, id, step, achievement_category
            FROM `event_achievements` 
            WHERE `account_id` = ?");
            $res->execute([$this->id]);
            $this->achievements = $res->fetchAll(\PDO::FETCH_UNIQUE | \PDO::FETCH_CLASS, Achievement::class);
        }
        return $this->achievements;
    }

    /**
     * @return Profession[]
     */
    public function getProfessions()
    {
        if (!$this->professions) {
            $res = \Core::getConnection()->prepare("SELECT profession_id, id 
            FROM `event_professions`
            WHERE `account_id` = ?");
            $res->execute([$this->id]);
            $this->professions = $res->fetchAll(\PDO::FETCH_UNIQUE | \PDO::FETCH_CLASS, Profession::class);
        }
        return $this->professions;
    }

    /**
     * @param int $quest_id
     * @return Quest
     */
    public function findQuest(int $quest_id)
    {
        if (!isset($this->quests[$quest_id])) {
            $res = \Core::getConnection()->prepare("SELECT * 
            FROM `event_quests` 
            WHERE `quest_id` = ? AND `account_id` = ? LIMIT 1");
            $res->execute([$quest_id, $this->id]);
            $this->quests[$quest_id] = $res->fetchObject(Quest::class);
        }
        return $this->quests[$quest_id];
    }

    /**
     * @param int $achievement_id
     * @return Achievement
     */
    public function findAchievement(int $achievement_id)
    {
        if (!isset($this->achievements[$achievement_id])) {
            $res = \Core::getConnection()->prepare("SELECT * 
            FROM `event_achievements` 
            WHERE `achievement_id` = ? AND `account_id` = ? LIMIT 1");
            $res->execute([$achievement_id, $this->id]);
            $this->achievements[$achievement_id] = $res->fetchObject(Achievement::class);
        }
        return $this->achievements[$achievement_id];
    }

    /**
     * @param int $profession_id
     * @return Profession
     */
    public function findProfession(int $profession_id)
    {
        if (!isset($this->professions[$profession_id])) {
            $res = \Core::getConnection()->prepare("SELECT * FROM `event_professions` 
            WHERE `profession_id` = ? AND `account_id` = ? LIMIT 1");
            $res->execute([$profession_id, $this->id]);
            $this->professions[$profession_id] = $res->fetchObject(Profession::class);
        }
        return $this->professions[$profession_id];
    }

    /**
     * @param int $quest_id
     * @return Quest
     */
    public function addQuest(int $quest_id)
    {
        return new Quest($quest_id, $this->id);
    }

    /**
     * @param int $achievement_id
     * @param int $achievement_category
     * @return Achievement
     */
    public function addAchievement(int $achievement_id, int $achievement_category)
    {
        if(!$this->achievements[$achievement_id]) {
            return $this->achievements[$achievement_id] = new Achievement($achievement_id, $achievement_category, $this->id);
        }
        return $this->achievements[$achievement_id];
    }

    /**
     * @param int $profession_id
     * @return Profession
     */
    public function addProfession(int $profession_id)
    {
        return new Profession($profession_id, $this->id);
    }

    public function onCheckLevel($level)
    {
        $res = \Core::getConnection()->prepare('SELECT * FROM achievements WHERE name = ?');
        $res->execute(['check_level']);

        if ($achievements = $res->fetchAll(\PDO::FETCH_CLASS, \data\Achievement::class)) {
            /**
             * @var $achievement \data\Achievement
             */
            foreach ($achievements AS $achievement) {
                if (!$event_achievement = $this->findAchievement($achievement->id)) {
                    $event_achievement = $this->addAchievement($achievement->id, $achievement->parent_id);
                }

                $event_achievement->setConfigInt('point', $level);

                $step = $event_achievement->step;
                $steps = $achievement->getSteps();

                foreach ($steps AS $complexity => $data) {
                    if ($data['point'] > $level) {
                        break;
                    }

                    if ($complexity == $step) {
                        $event_achievement->step++;
                        $event_achievement->exp += $data['exp'];
                        \Flash::addSystemMessage(sprintf('Достижение "%s" выполнено %s', $achievement->title, $event_achievement->step));
                    }
                }
            }
        }
    }

    public function onCheckAccount($life)
    {
        $res = \Core::getConnection()->prepare('SELECT * FROM achievements WHERE name = ?');
        $res->execute(['check_account']);

        if ($achievements = $res->fetchAll(\PDO::FETCH_CLASS, \data\Achievement::class)) {
            /**
             * @var $achievement \data\Achievement
             */
            foreach ($achievements AS $achievement) {
                if (!$event_achievement = $this->findAchievement($achievement->id)) {
                    $event_achievement = $this->addAchievement($achievement->id, $achievement->parent_id);
                }

                $event_achievement->setConfigInt('point', $life);

                $step = $event_achievement->step;
                $steps = $achievement->getSteps();

                foreach ($steps AS $complexity => $data) {
                    if ((3600 * 24 *$data['point']) > $life) {
                        break;
                    }

                    if ($complexity == $step) {
                        $event_achievement->step++;
                        $event_achievement->exp += $data['exp'];
                        \Flash::addSystemMessage(sprintf('Достижение "%s" выполнено %s', $achievement->title, $event_achievement->step));
                    }
                }
            }
        }
    }
}