<?php

namespace core\router;

use core\Request;

class Route
{
    const PATH_VARIABLE_PATTERN = '#\{([^\/]+?)\}#';

    protected $pattern;
    protected $result;
    protected $methods = ['GET'];

    public function __construct(string $pattern, callable $result)
    {
        $this->pattern = $pattern;
        $this->result = $result;
    }

    /**
     * @return string
     */
    public function getPattern(): string
    {
        return $this->pattern;
    }

    /**
     * @return callable
     */
    public function getResult(): callable
    {
        return $this->result;
    }

    /**
     * @return array
     */
    public function getMethods(): array
    {
        return $this->methods;
    }

    /**
     * @param array $methods
     */
    public function setMethods(...$methods)
    {
        $this->methods = $methods;
    }

    public function hasExecuteRequest(Request $request)
    {
        return !!preg_match($this->getPregPatternUrl(), $request->getUrl()->getPath());
    }
    public function getPathVariables()
    {
        $vars = [];
        preg_match_all(self::PATH_VARIABLE_PATTERN, $this->getPattern(), $vars, PREG_PATTERN_ORDER);
        return $vars[1];
    }
    public function getPregPatternUrl()
    {
        // статичные участки пути
        $static_paths = preg_split(self::PATH_VARIABLE_PATTERN, $this->getPattern());

        // переменные участки пути, расположенные между статичных
        $vars_matches = $this->getPathVariablesMatches();

        // собираем регулярку для обработки пути
        $pattern = '';
        for ($i = 0; $i < count($static_paths); $i++) {
            $pattern .= preg_quote($static_paths[$i], '#'); // статичные участки экранируем

            if (array_key_exists($i, $vars_matches)) {
                // переменных частей должно быть на одну меньше, чем статичных? поэтому на всякий случай проверяем
                $pattern .= $vars_matches[$i];
            }
        }

        return '#^' . $pattern . '$#';
    }
    public function getPathVariablesMatches()
    {
        $matches = [];
        $vars = $this->getPathVariables();
        foreach ($vars AS $var_name) {
            switch ($var_name) {
                case '**':
                    $matches[] = '(.+)';
                    break;
                case '*':
                default:
                    $matches[] = '([^/]+)';
                    break;
            }
        }
        return $matches;
    }
    public function getRequestPathVariables(Request $request)
    {
        $variables = [];
        if (!preg_match($this->getPregPatternUrl(), $request->getUrl()->getPath(), $variables)) {
            throw new \Exception("Не удалось получить параметры из запроса");
        }

        array_splice($variables, 0, 1); // удаляем полное вхождение url. Нам же нужны только переменные из пути

        foreach ($variables AS &$var) {
            $var = urldecode($var);
        }

        $variables_mapping = $this->getPathVariables();
        foreach ($variables_mapping AS $index => $var_name) {
            if ($var_name == '*' || $var_name == '**') {
                continue;
            }

            $variables[$var_name] = $variables[$index];
        }
        return $variables;
    }
}