<?php

use data\Account;
use data\Achievement;
use data\Profession;
use data\Item;
use data\Building;
use data\Npc;
use data\profession\Recipe;

abstract class Core
{
    protected static $connection;

    protected static $accounts = [];
    protected static $professions = [];
    protected static $items = [];
    protected static $achievements = [];
    protected static $recipes = [];
    protected static $buildings = [];
    protected static $npcs = [];
    protected static $events = [];

    /**
     * @return PDO
     * @throws Exception
     */
    public static function getConnection() {
        if (!self::$connection) {
            $dsn = 'mysql:host=localhost;dbname=magic_craft';
            $username = 'root';
            $password = null;

            try {
                self::$connection = new \PDO($dsn, $username, $password, array(
                    \PDO::ATTR_ERRMODE => \PDO::ERRMODE_EXCEPTION,
                    \PDO::ATTR_DEFAULT_FETCH_MODE => \PDO::FETCH_ASSOC
                ));
            } catch (\PDOException $e) {
                throw new Exception($e->getMessage());
            }
        }
        return self::$connection;
    }

    /**
     * @return Account[]
     */
    public static function getAccounts()
    {
        if (!self::$accounts) {
            $res = self::getConnection()->query("SELECT id, id, login, craft_reputation
            FROM `accounts`");
            self::$accounts = $res->fetchAll(\PDO::FETCH_UNIQUE|\PDO::FETCH_CLASS, Account::class);
        }
        return self::$accounts;
    }

    /**
     * @return Profession[]
     */
    public static function getProfessions()
    {
        if (!self::$professions) {
            $res = self::getConnection()->query("SELECT id, id, title, icon FROM `professions`");
            self::$professions = $res->fetchAll(\PDO::FETCH_UNIQUE|\PDO::FETCH_CLASS, Profession::class);
        }
        return self::$professions;
    }

    /**
     * @return Account[]
     */
    public static function getItems()
    {
        if (!self::$items) {
            $res = self::getConnection()->query("SELECT id, id FROM `items`");
            self::$items = $res->fetchAll(\PDO::FETCH_UNIQUE|\PDO::FETCH_CLASS, Item::class);
        }
        return self::$items;
    }

    /**
     * @return Achievement[]
     */
    public static function getAchievements()
    {
        if (!self::$achievements) {
            $res = self::getConnection()->query("SELECT id, id, title, icon, description, parent_id 
            FROM `achievements` 
            WHERE parent_id IS NULL");
            self::$achievements = $res->fetchAll(\PDO::FETCH_UNIQUE|\PDO::FETCH_CLASS, Achievement::class);
        }
        return self::$achievements;
    }

    /**
     * @return Recipe[]
     */
    public static function getRecipes()
    {
        if (!self::$achievements) {
            $res = self::getConnection()->query("SELECT id, id, life, energy, exp, item_id, item_count
            FROM `profession_recipes`");
            self::$achievements = $res->fetchAll(\PDO::FETCH_UNIQUE|\PDO::FETCH_CLASS, Recipe::class);
        }
        return self::$achievements;
    }

    /**
     * @return Building[]
     */
    public static function getBuildings()
    {
        if (!self::$buildings) {
            $res = self::getConnection()->query("SELECT id, id, parent_id, name
            FROM `location_buildings`
            WHERE parent_id IS NULL");
            self::$buildings = $res->fetchAll(\PDO::FETCH_UNIQUE|\PDO::FETCH_CLASS, Building::class);
        }
        return self::$buildings;
    }

    /**
     * @return Npc[]
     */
    public static function getNpcs()
    {
        if (!self::$npcs) {
            $res = self::getConnection()->query("SELECT id, id, name
            FROM `location_npcs`
            WHERE building_id IS NULL");
            self::$npcs = $res->fetchAll(\PDO::FETCH_UNIQUE|\PDO::FETCH_CLASS, Npc::class);
        }
        return self::$npcs;
    }

    public static function getEvents()
    {
        if (!self::$events) {
            $res = self::getConnection()->prepare("SELECT *
            FROM `location_events`
            WHERE `time_end` > ?");
            $res->execute([time()]);
            self::$events = $res->fetchAll();
        }
        return self::$events;
    }

    /**
     * @param int $id
     * @return Account
     */
    public static function findAccount($id) {
        if(!isset(self::$accounts[$id])) {
            $res = self::getConnection()->prepare("SELECT * FROM `accounts` WHERE `id` = ? LIMIT 1");
            $res->execute([$id]);
            self::$accounts[$id] = $res->fetchObject(Account::class);
        }
        return self::$accounts[$id];
    }

    /**
     * @param int $id
     * @return Profession
     */
    public static function findProfession($id) {
        if(!isset(self::$professions[$id])) {
            $res = self::getConnection()->prepare("SELECT * FROM `professions` WHERE `id` = ? LIMIT 1");
            $res->execute([$id]);
            self::$professions[$id] = $res->fetchObject(Profession::class);
        }
        return self::$professions[$id];
    }

    /**
     * @param int $id
     * @return Item
     */
    public static function findItem($id) {
        if(!isset(self::$items[$id])) {
            $res = self::getConnection()->prepare("SELECT * FROM `items` WHERE `id` = ? LIMIT 1");
            $res->execute([$id]);
            self::$items[$id] = $res->fetchObject(Item::class);
        }
        return self::$items[$id];
    }

    /**
     * @param int $id
     * @return Achievement
     */
    public static function findAchievement($id) {
        if(!isset(self::$achievements[$id])) {
            $res = self::getConnection()->prepare("SELECT * FROM `achievements` 
            WHERE `id` = ? AND parent_id IS NULL LIMIT 1");
            $res->execute([$id]);
            self::$achievements[$id] = $res->fetchObject(Achievement::class);
        }
        return self::$achievements[$id];
    }

    /**
     * @param int $id
     * @return Recipe
     */
    public static function findRecipe($id) {
        if(!isset(self::$recipes[$id])) {
            $res = self::getConnection()->prepare("SELECT * FROM `profession_recipes` 
            WHERE `id` = ? LIMIT 1");
            $res->execute([$id]);
            self::$recipes[$id] = $res->fetchObject(Recipe::class);
        }
        return self::$recipes[$id];
    }

    /**
     * @param string $name
     * @return Building
     */
    public static function findBuilding($name) {
        if(!isset(self::$buildings[$name])) {
            $res = self::getConnection()->prepare("SELECT * FROM `location_buildings` 
            WHERE `name` = ? AND parent_id IS NULL LIMIT 1");
            $res->execute([$name]);
            self::$buildings[$name] = $res->fetchObject(Building::class);
        }
        return self::$buildings[$name];
    }

    /**
     * @param int $id
     * @return Npc
     */
    public static function findNpc($id) {
        if(!isset(self::$npcs[$id])) {
            $res = self::getConnection()->prepare("SELECT * FROM `location_npcs` 
            WHERE `id` = ? AND building_id IS NULL LIMIT 1");
            $res->execute([$id]);
            self::$npcs[$id] = $res->fetchObject(Npc::class);
        }
        return self::$npcs[$id];
    }

    /**
     * @param int $id
     * @return array
     */
    public static function findEvent($id) {
        if(!isset(self::$events[$id])) {
            $res = self::getConnection()->prepare("SELECT * FROM `location_events` 
            WHERE `id` = ? AND time_end > ? LIMIT 1");
            $res->execute([$id, time()]);
            self::$events[$id] = $res->fetch();
        }
        return self::$events[$id];
    }
}