<?php
class Guilds extends CActiveRecord {

    public $text;
    public $price = 100;
    
    public function tableName() {
        return 'guilds';
    }

    public function rules() {
        return array(
            array('level, iron, xp, money', 'numerical', 'integerOnly' => true),
            array('money', 'checkMoney', 'on' => 'insert'),
            array('name', 'unique', 'className' => 'Guilds', 'caseSensitive' => false),
            array('name', 'match', 'pattern' => '/^[а-яА-ЯёЁa-zA-Z][а-яА-ЯёЁa-zA-Z0-9-_\s]{2,16}$/u', 'allowEmpty'=>false, 'message' => 'Название гильдии содержит запрещённые символы'),
            array('name', 'length', 'min' => 3, 'max' => 16, 'tooShort' => 'Название гильдии должно быть длиной от 3 до 16 символов', 'tooLong' => 'Название гильдии должно быть длиной от 3 до 16 символов'),
            array('text', 'length', 'max' => 64, 'tooLong' => 'Объявление гильдии должно быть больше 64 символов'),
            array('id_guild, logo, created, side', 'safe'),
        );
    }

    public function checkMoney($attribute,$params) {
        if($this->$attribute < $this->price*100)
            $this->addError('name',Yii::t('layout','У Вас недостаточно золота чтобы создать гильдию'));
    }
        
    public function relations() {
        return array(
            'users' => array(self::HAS_MANY, 'Users', 'guild_id', 'order'=>'users.guild_rank DESC, users.guild_xp DESC'),
            'buildings' => array(self::HAS_MANY, 'GuildsBuildings', 'id_guild'),
        );
    }

    public function behaviors() {
        return array('CAdvancedArBehavior' => array(
            'class' => 'application.extensions.CAdvancedArBehavior.CAdvancedArBehavior'));
    }
    
    public function attributeLabels() {
        return array(
            'id_guild' => 'Id Guild',
            'name' => 'Название гильдии:',
            'info' => 'Новое объявление:',
            'text' => 'Новое объявление:',
            'level' => 'Уровень',
            'xp' => 'Опыт',
            'money' => 'Золото',
            'iron' => 'Железо',
            'rank_invite' => 'Приглашать в гильдию может:',
            'rank_kick' => 'Выгонять из гильдии может:',
        );
    }

    protected function beforeSave() {
        if (parent::beforeSave()) {
            if ($this->isNewRecord) {
                $this->created = time();
            }
            return true;
        } else {
            return false;
        }
    }
    
    /**
     * Проверяет состоит ли человек в клане и есть ли у него новые объявления
     * @return boolean
     */
    public static function checkGuildInfo() {
        if (!Yii::app()->user->isGuest) {
            $user = Users::checkForGuild();
            if ($user && Yii::app()->controller->id != 'guild') {
                $criteria = new CDbCriteria;
                $criteria->select = array('info', 'info_time', 'info_creator');
                if (Yii::app()->request->cookies['guild_info_time'])
                    $criteria->addCondition('info_time > '.Yii::app()->request->cookies['guild_info_time']);
                $criteria->compare('id_guild', $user->guild_id);
                return self::model()->find($criteria); 
            } 
        }
        return false;
    }
    
    /**
     * Выводим название клана
     * @param type $id_guild
     * @return null
     */
    public static function getName($id_guild) {
        $guild = self::model()->findByPk($id_guild);
        return '<img src="/images/icons/guild_'.$guild->side.'.png" alt="*"> '.$guild->name;
    }
    
    public function search() {
        $criteria = new CDbCriteria;

        $criteria->compare('id_guild', $this->id_guild);
        $criteria->compare('name', $this->name, true);
        $criteria->compare('info', $this->info, true);
        $criteria->compare('level', $this->level);
        $criteria->compare('xp', $this->xp);
        $criteria->compare('money', $this->money);
        $criteria->compare('iron', $this->iron);
        $criteria->compare('logo', $this->logo);

        return new CActiveDataProvider($this, array(
            'criteria' => $criteria,
        ));
    }

    public static function model($className = __CLASS__) {
        return parent::model($className);
    }
}
