<?php

/**
 * This is the model class for table "cities".
 *
 * The followings are the available columns in table 'cities':
 * @property string $city_id
 * @property string $country_id
 * @property string $region_id
 * @property string $name
 *
 * The followings are the available model relations:
 * @property Regions $region
 * @property Countries $country
 * @property UserEnterprises[] $userEnterprises
 */
class Cities extends CActiveRecord
{
	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return 'cities';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		// NOTE: you should only define rules for those attributes that
		// will receive user inputs.
		return array(
			array('country_id', 'length', 'max'=>11),
			array('region_id', 'length', 'max'=>10),
			array('name', 'length', 'max'=>128),
			// The following rule is used by search().
			// @todo Please remove those attributes that should not be searched.
			array('city_id, country_id, region_id, name', 'safe', 'on'=>'search'),
		);
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		// NOTE: you may need to adjust the relation name and the related
		// class name for the relations automatically generated below.
		return array(
			'region' => array(self::BELONGS_TO, 'Regions', 'region_id'),
			'country' => array(self::BELONGS_TO, 'Countries', 'country_id'),
			'userEnterprises' => array(self::HAS_MANY, 'UserEnterprises', 'city_id'),
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'city_id' => 'City',
			'country_id' => 'Country',
			'region_id' => 'Region',
			'name' => 'Name',
		);
	}

	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 *
	 * Typical usecase:
	 * - Initialize the model fields with values from filter form.
	 * - Execute this method to get CActiveDataProvider instance which will filter
	 * models according to data in model fields.
	 * - Pass data provider to CGridView, CListView or any similar widget.
	 *
	 * @return CActiveDataProvider the data provider that can return the models
	 * based on the search/filter conditions.
	 */
	public function search()
	{
		// @todo Please modify the following code to remove attributes that should not be searched.

		$criteria=new CDbCriteria;

		$criteria->compare('city_id',$this->city_id);
		$criteria->compare('country_id',$this->country_id);
		$criteria->compare('region_id',$this->region_id);
		$criteria->compare('name',$this->name,true);

		return new CActiveDataProvider($this, array(
			'criteria'=>$criteria,
		));
	}

	/**
	 * Returns the static model of the specified AR class.
	 * Please note that you should have this exact method in all your CActiveRecord descendants!
	 * @param string $className active record class name.
	 * @return Cities the static model class
	 */
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}
        
    /**
     * 
     * @param array $user_ids
     * @return type
     */    
        public static function findAllByUsers(array $user_ids)
    {
        $criteria = new CDbCriteria();
        //$criteria->with = array('userEnterprises.userDepartments.userPositions.users');        
        //$criteria->addInCondition('users.id_user', $user_ids);
        
        return self::model()->findAll($criteria);        
    }
    
      public static function findAllByUsersAndCountry(array $user_ids, $country_id)
    {
        $criteria = new CDbCriteria();
        $criteria->with = array('userEnterprises.userDepartments.userPositions.users');        
        $criteria->addInCondition('users.id_user', $user_ids);
        $criteria->compare('country_id', $country_id);
        
        return self::model()->findAll($criteria);        
    }
    
      public static function findAllByUsersAndRegion(array $user_ids, $region_id)
    {
        $criteria = new CDbCriteria();
        $criteria->with = array('userEnterprises.userDepartments.userPositions.users');        
        $criteria->addInCondition('users.id_user', $user_ids);
        $criteria->compare('region_id', $region_id);
        
        return self::model()->findAll($criteria);        
    }
    
    /**
     * Получает список городов, которым принадлежат переданные предприятия.
     * @param array $enterprise_ids
     * @return CActiveRecord
     */
    
    public static function findAllByEnterprises(array $enterprise_ids)
    {
        $criteria = new CDbCriteria();
        $criteria->with = array('userEnterprises');                
        $criteria->addInCondition('userEnterprises.id_ent', $enterprise_ids);
        
        return self::model()->findAll($criteria);
    }
    
    /**
     * Получает список городов переданной страны, которым принадлежат переданные предприятия.
     * @param array $enterprise_ids
     * @param type $country_id
     * @return CActiveRecord
     */
    public static function findAllByEnterprisesAndCountry(array $enterprise_ids, $country_id){
        $criteria = new CDbCriteria();
        $criteria->with = array('userEnterprises');                
        $criteria->addInCondition('userEnterprises.id_ent', $enterprise_ids);
        $criteria->compare('country_id', $country_id);
        
        return self::model()->findAll($criteria);                
    }
    
    /**
     * Получает список городов переданного региона, которому принадлежат переданные предприятия.
     * @param array $enterprise_ids
     * @param type $country_id
     * @return CActiveRecord
     */
    public static function findAllByEnterprisesAndRegion(array $enterprise_ids, $region_id){
        $criteria = new CDbCriteria();
        $criteria->with = array('userEnterprises');                
        $criteria->addInCondition('userEnterprises.id_ent', $enterprise_ids);
        $criteria->compare('region_id', $region_id);
        
        return self::model()->findAll($criteria);                
    }
    
    public static function findAllByRegion($id_reg=null) {
        $criteria = new CDbCriteria();
                
        $criteria->compare('region_id', $id_reg);
        return self::model()->findAll($criteria);
    }
    
    public static function getDropDownOptions()
    {
        $cities = self::model()->findAll();
        $options = "";
        foreach($cities as $city)
        {
            $options.='<option value="'.$city->city_id.'">' . $city->name . '</option>';
        }
            return $options;
    }
}
