<?php
/**
 * Created by PhpStorm.
 * User: pavlovd
 * Date: 19.09.2015
 * Time: 17:04
 */

namespace Dcms\System\Packages;


use Dcms\Helpers\Json;
use Dcms\System\Core;
use Dcms\System\Packages\Models\Package;

abstract class Packages
{
    protected static $_packages = null;
    protected static $_instances = [];

    function __construct()
    {
        $this->update();
        $this->_loadPackages();
    }

    protected function _loadPackages()
    {
        $db = Core::getDb();
        if (is_null(self::$_packages)) {
            try {
                self::$_packages = $db->query("SELECT * FROM `packages`")->fetchAll();
            } catch (\Exception $e) {
                self::$_packages = [];
            }
        }
    }

    public function update()
    {
        $prep = Core::getDb()->prepare('INSERT INTO `packages`
(`type`, `name`, `version`, `installed`, `enabled`, `system`, `config_json`)
VALUES (:type, :name, :version, :installed, :enabled, :system, :config_json)
ON DUPLICATE KEY UPDATE `version` = VALUES(`version`), `system` = VALUES(`system`), `config_json` = VALUES(`config_json`)');
        $packages = (array)@glob($this->getPackagesPath() . '/*', GLOB_ONLYDIR);
        foreach ($packages AS $package_path) {
            if (!file_exists($package_path . '/package.json')) {
                continue;
            }

            $config_json = @file_get_contents($package_path . '/package.json');
            $config = Json::parse($config_json);
            $this->_checkConfig($config);

            $prep->execute([
                ':type' => $this->getType(),
                ':name' => $config['name'],
                ':version' => $config['version'],
                ':installed' => 1,
                ':enabled' => 1,
                ':system' => 1,
                ':config_json' => Json::stringify($config, false)
            ]);
        }
        self::$_packages = null;
    }

    protected function _checkConfig($config)
    {
        if (!array_key_exists('type', $config)) {
            throw new \Exception(__("Не указан тип пакета\n%s", Json::stringify($config)));
        }

        if ($config['type'] !== $this->getType()) {
            throw new \Exception(__("Тип пакета должен быть %s (указан %s)\n%s", $this->getType(), $config['type'],
                Json::stringify($config)));
        }

        if (!array_key_exists('name', $config)) {
            throw new \Exception(__("Не указано имя пакета\n%s", Json::stringify($config)));
        }

        if (!array_key_exists('version', $config)) {
            throw new \Exception(__("Не указана версия пакета\n%s", Json::stringify($config)));
        }
    }

    /**
     * Возвращает список пакетов, зарегистрированных в системе
     * @return \string[]
     */
    protected function getList()
    {
        $type = $this->getType();
        $list = [];
        foreach (self::$_packages AS $package) {
            if ($package['type'] === $type) {
                $list[] = $package['name'];
            }
        }
        return $list;
    }

    /**
     * Проверяет на наличие пакета
     * @param $name
     * @return bool
     */
    public function hasPackage($name)
    {
        return in_array($name, $this->getList());
    }

    /**
     * Возвращает информацию о пакете (запись таблицы packages)
     * @param string $name
     * @return array
     * @throws \Exception
     */
    public function getData($name)
    {
        $type = $this->getType();
        if (!$this->hasPackage($name)) {
            throw new \Exception(__("Пакет %s не найден", $name));
        }

        foreach (self::$_packages AS $package) {
            if ($package['name'] === $name && $package['type'] === $type) {
                return $package;
            }
        }

        throw new \Exception(__("Пакет %s не найден (что крайне странно)", $name));
    }

    /**
     * Возвращает конфиг пакета (package.json)
     * @param string $name
     * @return array mixed
     * @throws \Exception
     */
    public function getConfig($name)
    {
        return Json::parse($this->getData($name)['config_json']);
    }

    /**
     * Возвращает версию пакета
     * @param $name
     * @return string
     * @throws \Exception
     */
    public function getVersion($name)
    {
        return (string)$this->getData($name)['version'];
    }

    /**
     * Флаг, установлен ли пакет
     * @param $name
     * @return bool
     * @throws \Exception
     */
    public function getIsInstalled($name)
    {
        return (bool)$this->getData($name)['installed'];
    }

    /**
     * Флаг, включен ли пакет
     * @param $name
     * @return bool
     * @throws \Exception
     */
    public function getIsEnabled($name)
    {
        return (bool)$this->getData($name)['enabled'];
    }

    /**
     * Флаг, системный ли пакет
     * @param $name
     * @return bool
     * @throws \Exception
     */
    public function getIsSystem($name)
    {
        return (bool)$this->getData($name)['system'];
    }

    /**
     * Возвращает тип пакета (plugin, theme, widget, module)
     * @return string
     */
    abstract public function getType();

    /**
     * Возвращает имя класса модуля с пространством имен по имени модуля
     * @param string $name
     * @return string
     */
    abstract protected function _getClassName($name);

    /**
     * Путь к директории с пакетами
     * @return string
     */
    abstract public function getPackagesPath();

    /**
     * Возвращает инстанс пакета
     * @param string $name
     * @return Package
     */
    protected function _getInstance($name)
    {
        $type = $this->getType();
        $key = $type . '|' . $name;
        if (!array_key_exists($key, self::$_instances)) {
            $class_name = $this->_getClassName($name);
            $instance = new $class_name($type, $name);
            // TODO: сделать проверку на тип класса
            self::$_instances[$key] = $instance;
        }
        return self::$_instances[$key];
    }

    /**
     * Запускает процесс установки (изменение или создание таблиц, записей)
     * @param string $name
     * @throws \Exception
     */
    public function install($name)
    {
        if (!$this->hasPackage($name)) {
            throw new \Exception(__("Пакет %s не зарегистрирован", $name));
        }

        if ($this->getIsInstalled($name)) {
            throw new \Exception(__("Пакет %s уже установлен", $name));
        }

        $instance = $this->_getInstance($name);
        $instance->onInstalling();

        Core::getDb()->prepare("UPDATE `packages` SET `installed` = 1 WHERE `name` = :name AND `type` = :type")->execute(
            [
                ':name' => $name,
                ':type' => $this->getType($name)
            ]
        );
        self::$_packages = null;
    }

    /**
     * Включает пакет
     * @param string $name
     * @throws \Exception
     */
    public function enable($name)
    {
        if (!$this->hasPackage($name)) {
            throw new \Exception(__("Пакет %s не зарегистрирован", $name));
        }

        if (!$this->getIsInstalled($name)) {
            throw new \Exception(__("Пакет %s не установлен", $name));
        }

        if ($this->getIsEnabled($name)) {
            throw new \Exception(__("Пакет %s уже активен", $name));
        }


        $instance = $this->_getInstance($name);
        $instance->onEnabling();

        Core::getDb()->prepare("UPDATE `packages` SET `enabled` = 1 WHERE `name` = :name AND `type` = :type")->execute(
            [
                ':name' => $name,
                ':type' => $this->getType($name)
            ]
        );
        self::$_packages = null;
    }

    /**
     * Выключает пакет
     * @param string $name
     * @throws \Exception
     */
    public function disable($name)
    {
        if (!$this->hasPackage($name)) {
            throw new \Exception(__("Пакет %s не зарегистрирован", $name));
        }

        if (!$this->getIsInstalled($name)) {
            throw new \Exception(__("Пакет %s не установлен", $name));
        }

        if (!$this->getIsEnabled($name)) {
            throw new \Exception(__("Пакет %s не активен", $name));
        }


        $instance = $this->_getInstance($name);
        $instance->onDisabling();

        Core::getDb()->prepare("UPDATE `packages` SET `enabled` = 0 WHERE `name` = :name AND `type` = :type")->execute(
            [
                ':name' => $name,
                ':type' => $this->getType($name)
            ]
        );
        self::$_packages = null;
    }

    /**
     * Удаляет пакет (возвращает систему в исходное состояние)
     * @param string $name
     * @throws \Exception
     */
    public function deinstall($name)
    {

        if (!$this->hasPackage($name)) {
            throw new \Exception(__("Пакет %s не зарегистрирован", $name));
        }

        if (!$this->getIsInstalled($name)) {
            throw new \Exception(__("Пакет %s не установлен", $name));
        }

        if ($this->getIsEnabled($name)) {
            $this->disable($name);
        }


        $instance = $this->_getInstance($name);
        $instance->onRemoving();

        Core::getDb()->prepare("UPDATE `packages` SET `installed` = 0 WHERE `name` = :name AND `type` = :type")->execute(
            [
                ':name' => $name,
                ':type' => $this->getType($name)
            ]
        );
        self::$_packages = null;
    }
}