<?php
/**
 * Created by PhpStorm.
 * User: pavlovd
 * Date: 24.09.2015
 * Time: 17:34
 */

namespace Dcms\System\Packages\Models;


use Dcms\Helpers\Configurator;
use Dcms\System\Request;

/**
 * Привязка метода модуля к Url.
 * Проверяет, может ли метод обработать запрос, а также возвращает именованные значения переменных из Url
 * Class ModuleMap
 * @package Dcms\System\Packages\Models
 */
class ModuleMap extends Configurator
{
    const PATH_VARIABLE_PATTERN = '#\{([^\/]+?)\}#';


    protected
        $_method = "",
        $_url = "",
        $_execute = "",
        $_module_name;

    /**
     * HTTP метод
     * @return string
     */
    public function getMethod()
    {
        return $this->_method;
    }

    /**
     * HTTP метод
     * @param string $method
     */
    public function setMethod($method)
    {
        $this->_method = $method;
    }

    /**
     * Шаблон пути
     * @return string
     */
    public function getUrl()
    {
        return $this->_url;
    }

    /**
     * Шаблон пути
     * @param string $url
     */
    public function setUrl($url)
    {
        $this->_url = $url;
    }

    /**
     * Метод класса для обработки запроса
     * @return string
     */
    public function getExecute()
    {
        return $this->_execute;
    }

    /**
     * Метод класса для обработки запроса
     * @param string $execute
     */
    public function setExecute($execute)
    {
        $this->_execute = $execute;
    }

    /**
     * Возвращает имена переменных в пути запроса
     * @return string[]
     */
    public function getPathVariables()
    {
        $vars = [];
        preg_match_all(self::PATH_VARIABLE_PATTERN, $this->getUrl(), $vars, PREG_PATTERN_ORDER);
        return $vars[1];
    }

    /**
     * Возвращает регулярки для обработки переменных в пути запроса
     * @return string[]
     */
    public function getPathVariablesMatches()
    {
        $matches = [];
        $vars = $this->getPathVariables();
        foreach ($vars AS $var_name) {
            switch ($var_name) {
                case '**':
                    $matches[] = '(.+)';
                    break;
                case '*':
                default:
                    $matches[] = '([^/]+)';
                    break;
            }
        }
        return $matches;
    }

    /**
     * Возвращает регулярку для обработки пути
     * @return string
     */
    public function getPregPatternUrl()
    {
        // статичные участки пути
        $static_paths = preg_split(self::PATH_VARIABLE_PATTERN, $this->getUrl());
        // переменные участки пути, расположенные между статичных
        $vars_matches = $this->getPathVariablesMatches();

        // собираем регулярку для обработки пути
        $pattern = '';
        for ($i = 0; $i < count($static_paths); $i++) {
            $pattern .= preg_quote($static_paths[$i], '#'); // статичные участки экранируем

            if (array_key_exists($i, $vars_matches)) {
                // переменных частей должно быть на одну меньше, чем статичных? поэтому на всякий случай проверяем
                $pattern .= $vars_matches[$i];
            }
        }

        return '#^' . $pattern . '$#';
    }

    /**
     * Проверяем, можем ли мы обработать данный запрос
     * @param Request $request
     * @return bool
     */
    public function hasExecuteRequest(Request $request)
    {
        if ($request->getMethod() !== $this->getMethod()) {
            return false;
        }

        return !!preg_match($this->getPregPatternUrl(), $request->getUrl()->getPath());
    }

    /**
     * Переменные пути запроса
     * @param Request $request
     * @return array
     * @throws \Exception
     */
    public function getRequestPathVariables(Request $request)
    {
        $variables = [];
        if (!preg_match($this->getPregPatternUrl(), $request->getUrl()->getPath(), $variables)) {
            throw new \Exception(__("Не удалось получить параметры из запроса"));
        }

        array_splice($variables, 0, 1); // удаляем полное вхождение url. Нам же нужны только переменные из пути

        foreach($variables AS &$var){
            $var = urldecode($var);
        }

        $variables_mapping = $this->getPathVariables();
        foreach ($variables_mapping AS $index => $var_name) {
            if ($var_name == '*' || $var_name == '**') {
                continue;
            }

            $variables[$var_name] = $variables[$index];
        }
        return $variables;
    }

    /**
     * @return string
     */
    public function getModuleName()
    {
        return $this->_module_name;
    }

    /**
     * @param string $module_name
     */
    public function setModuleName($module_name)
    {
        $this->_module_name = $module_name;
    }

    /**
     * Возвращает индекс совпадения запроса с путем
     * @param Request $request
     * @return int
     */
    public function getCompareIndex(Request $request)
    {
        $deep = substr_count($this->getUrl(), '/');
        $vars = $this->getPathVariables();
        $vars_count = count($vars);
        $static_path_deep = $deep - $vars_count;
        $greedy = in_array('**', $vars);

        return $static_path_deep + ($greedy ? 0 : $vars_count * 0.5);
    }

}