<?php
/**
 * Created by PhpStorm.
 * User: pavlovd
 * Date: 19.09.2015
 * Time: 15:01
 */

namespace Dcms\Helpers;

use Dcms\Ui\Ui;

/**
 * Класс позволяет инициализировать объект класса с передачей произвольных параметров в конструктор в виде ассоциативного массива.
 * Например: Если у класса ClassName есть метод setParamOne, то его можно инициализировать как new ClassName(["ParamOne" => "ValueParamOne"])
 * Также у объекта можно получить весь список параметров в виде ассоциативного массива при помощи метода getConfig
 * Class Configurator
 * @package Dcms\Helpers
 */
abstract class Configurator
{
    protected $_data = [];

    public function __construct($config = [])
    {
        if ($config) {
            $this->setConfig($config);
        }
    }

    /**
     * параметры, которые не должны попасть в конфиг
     * @return array
     */
    protected function _getSkippedConfigParams()
    {
        return ['Id', 'Config'];
    }

    /**
     * Установка конфига
     * @param array $config
     * @throws \Exception
     */
    final public function setConfig($config)
    {
        $skipped = $this->_getSkippedConfigParams();
        foreach ($config AS $param_name => $param_value) {
            if (in_array($param_name, $skipped)) {
                continue;
            }

            $method_name = 'set' . $param_name;
            if (!method_exists($this, $method_name)) {
                $class_name = get_class($this);
                throw new \Exception('Метод ' . $method_name . ' отсутствует у класса ' . $class_name);
            }
            $this->$method_name($param_value);
        }
    }

    /**
     * Получение конфига объекта
     * @return array
     */
    final public function getConfig()
    {
        $config = [];
        $skipped = $this->_getSkippedConfigParams();
        $methods = get_class_methods($this);
        foreach ($methods AS $method_name) {
            if (strpos($method_name, 'get') !== 0) {
                continue;
            }
            $param_name = substr_replace($method_name, '', 0, 3);
            if (in_array($param_name, $skipped)) {
                continue;
            }

            $param_value = $this->$method_name();

            if ($param_value && $param_value instanceof Configurator) {
                $param_value = $param_value->getConfig();

            }
            if ($param_value && $param_value instanceof Url) {
                $param_value = (string)$param_value;
            }

            if ($param_value && is_array($param_value)) {
                foreach ($param_value AS $param_value_index => $param_value_item) {
                    if ($param_value_item && $param_value_item instanceof Configurator) {
                        $param_value[$param_value_index] = $param_value_item->getConfig();
                    }
                    if ($param_value_item && $param_value_item instanceof Url) {
                        $param_value[$param_value_index] = (string)$param_value_item;
                    }
                }
            }

            $config[$param_name] = $param_value;
        }
        return $config;
    }

    /**
     * Пользовательские данные, которые могут быть использованы по любому назначению
     * @return array
     */
    final public function getData()
    {
        return $this->_data;
    }

    /**
     * Пользовательские данные, которые могут быть использованы по любому назначению
     * @param array $data
     */
    final public function setData($data)
    {
        $this->_data = $data;
    }
}