<?php
/**
 * Created by PhpStorm.
 * User: pavlovd
 * Date: 19.09.2015
 * Time: 15:05
 */

namespace Dcms\System\Packages;


use Dcms\Helpers\FileSystem;
use Dcms\System\Packages\Models\Theme;
use Dcms\System\Taxonomy;

class Themes extends Packages
{

    /**
     * Возвращает тип пакета (plugin, theme, widget, module)
     * @return string
     */
    public function getType()
    {
        return 'theme';
    }

    public function getPackagesPath()
    {
        return H . '/sys/packages/Themes';
    }

    /**
     * Возвращает имя класса модуля с пространством имен по имени модуля
     * @param string $name
     * @return string
     */
    protected function _getClassName($name)
    {
        return '\\Themes\\' . $name . '\\Theme';
    }

    /**
     * Возвращает путь к шаблону по имени класса для указанной темы
     * @param string $theme_name Название темы оформления
     * @param string $class_name Название класса
     * @return string
     * @throws \Exception
     */
    public function getThemeTemplatePath($theme_name, $class_name)
    {
        do {
            $templates_path = $this->getThemeTemplatesPath($theme_name);
            if ($templates_path) {
                $template_path = $templates_path . '/' . $class_name . '.php';
                if ($path = realpath($template_path)) {
                    return $path;
                }
            }
        } while ($theme_name = $this->getThemeParent($theme_name));

        throw new \Exception(__("Тема %s не может отрисовать модель %s", $theme_name, $class_name));
    }

    /**
     * Возвращает путь к папке с шаблонами
     * @param $theme_name
     * @return bool|string
     */
    public function getThemeTemplatesPath($theme_name)
    {
        $config = $this->getConfig($theme_name);
        $tpl_dir = $config['templates_path'];

        return realpath(H . '/sys/packages/Themes/' . $theme_name . '/' . $tpl_dir);
    }

    /**
     * Поиск ресурсов темы оформления
     * @param string $theme_name Имя темы оформления
     * @param string $res_name Ключ ресурса (js, css)
     * @return string[] Массив относительных путей к файлам ресурсов
     */
    protected function _searchThemeResources($theme_name, $res_name)
    {
        $config = $this->getConfig($theme_name);
        $theme_path = realpath($this->getPackagesPath() . '/' . $theme_name);

        if (!array_key_exists('resources', $config) || !array_key_exists($res_name, $config['resources'])) {
            return [];
        }

        $all_files_rel_path = [];
        $patterns = (array)$config['resources'][$res_name];
        foreach ($patterns AS $pattern) {
            $files_abs_path = (array)glob($theme_path . '/' . $pattern, GLOB_NOSORT);
            foreach ($files_abs_path AS $file_abs_path) {
                $all_files_rel_path[] = FileSystem::getRelPath($file_abs_path);
            }
        }

        return array_unique($all_files_rel_path);
    }

    /**
     * Поиск ресурсов темы оформления, рекурсивно проходя все зависимые темы
     * @param string $theme_name Имя темы оформления
     * @param string $res_name Ключ ресурса (js, css)
     * @return string[] Массив относительных путей к файлам ресурсов
     */
    public function getThemeResources($theme_name, $res_name)
    {
        $resources = $this->_searchThemeResources($theme_name, $res_name);
        while ($theme_name = $this->getThemeParent($theme_name)) {
            $resources = array_merge($this->getThemeResources($theme_name, $res_name), $resources);
        }
        return array_unique($resources);
    }

    /**
     * Возвращает название темы-родителя
     * Если тема является системной, то возвращает false
     * @param string $theme_name
     * @return false|string
     * @throws \Exception
     */
    public function getThemeParent($theme_name)
    {
        $config = $this->getConfig($theme_name);

        if ($this->getIsSystem($theme_name))
            return false;

        if ($this->hasPackage($config['parent'])) {
            return $config['parent'];
        }

        throw new \Exception(__("Не найдена родительская тема (%s) у темы (%s)", $config['parent'], $theme_name));
    }


    /**
     * Возвращает тему оформления по-умолчанию.
     * Берется первая попавшаяся активная тема оформления
     * @return string
     */
    private function _getDefaultTheme()
    {
        $all_themes = $this->getList();
        foreach ($all_themes AS $theme_name) {
            if ($this->getIsEnabled($theme_name)) {
                return $theme_name;
            }
        }
        return $all_themes[0];
    }

    /**
     * Возвращает текущую тему оформления
     * @return string
     * @throws \Exception
     */
    public function getCurrentTheme()
    {
        $current_theme = 'System';

        if (!$current_theme || !$this->hasPackage($current_theme) || !$this->getIsEnabled($current_theme)) {
            $current_theme = $this->_getDefaultTheme();
        }

        if (!$current_theme || !$this->hasPackage($current_theme) || !$this->getIsEnabled($current_theme)) {
            throw new \Exception(__("Не найдено ни одной активной темы оформления"));
        }

        return $current_theme;
    }

    /**
     * @param $theme_name
     * @param Taxonomy $taxonomy
     * @return string
     * @internal param $menu_name
     */
    public function renderThemeMenu($theme_name, Taxonomy $taxonomy)
    {
        /**
         * @var $inst Theme
         */
        $inst = $this->_getInstance($theme_name);
        return $inst->renderMenu($taxonomy);
    }
}