<?php
    /**
     * @class Validator
     * @author Patsifist
     * @version 0.1
     */

    Class Validator
    {
        public $attributes, $errors, $labels;

        /**
         * @param array $attributes массив полей
         * @param array $labels массив названий полей
         */

        public function __construct(array $attributes, $labels = array())
        {
            $this->attributes = $attributes;
            $this->labels = $labels;
        }


        /**
         * @param array $rules массив правил для проверки
         * @return bool Возаращает TRUE, если ошибок нет
         */

        public function check(array $rules)
        {
            foreach ($rules as $rule)
            {
                call_user_func(
                    array($this, 'validator_'.$rule[1]), // вызов метода / $validator->validator_require и тд
                    $rule[0], // название поля, которое проверяется
                    (isset($this->attributes[$rule[0]]) ? $this->attributes[$rule[0]] : false), // содержимое  поля, которое проверяется
                    (isset($rule[2]) ? $rule[2] : false) // дополнительный параметр
                    );
            }

            return empty($this->errors);
        }


        /**
         * @param string $attribute поле для проверки
         * Можно указать несколько полей, указав их черех запятую: array('name, password', 'require');
         */

        public function validator_require($attribute)
        {
            $attributes = explode(', ', $attribute);
            foreach($attributes as $attr)
            {
                if (empty($this->attributes[$attr]))
                {
                    $this->errors[] = 'Поле "'.$this->labels[$attr]. '" не заполнено';
                }
            }
        }



        /**
         * @param string $attribute поле для проверки
         * @param string $value содержимое поля для проверки
         * @param string $var2 переменная для сравнения с $value, можно быть названием поля: array('password', 'compare', 'password3');
         * @return bool
         */

        public function validator_compare($attribute, $value, $var2)
        {
            if (isset($this->attributes[$var2]))
            {
                if ($value != $this->attributes[$var2])
                {
                    $this->errors[] = $this->labels[$attribute] . ' и ' . $this->labels[$var2] . ' должны быть одинаковыми';
                    return false;
                }
            }
            elseif ($value != $var2)
            {
                $this->errors[] = $this->labels[$attribute] . ' не соответствует ' . $var2;
                return false;
            }
        }



        /**
         * @param string $attribute поле для проверки
         * @param string $value содержимое поля для проверки
         * @param string $type Разрешенный тип данных: array('age', 'type', 'integer');
         * @return bool
         */

        public function validator_type($attribute, $value, $type)
        {
            $types = array('boolean' => 'булевая', 'integer' => 'число', 'array' => 'массив', 'string' => 'строка', 'float' => 'дробное число', 'object' => 'объект');
            switch($type)
            {
            default:
            case 'string':
                $func = 'is_string';
                break;
            case 'boolean':
                $func = 'is_bool';
                break;
            case 'integer':
                $func = 'is_numeric';
                break;
            case 'array':
                $func = 'is_array';
                break;
            case 'float':
                $func = 'is_float';
                break;
            case 'object':
                $func = 'is_object';
                break;
            }

            $attributes = explode(', ', $attribute);
            foreach($attributes as $attr)
            {
                if (!call_user_func($func, $this->attributes[$attr]))//gettype($this->attributes[$attr]) != $type)
                {
                    $this->errors[] = $this->labels[$attr] . ' не ' . $types[$type];
                    return false;
                }
            }
        }



        /**
         * @param string $attribute поле для проверки
         * @param string $value содержимое поля для проверки
         * @param array $list список возможных значений: array('city', 'in_list', array('London', 'Paris'))
         * @return bool
         */

        public function validator_in_list($attribute, $value, $list)
        {
            /*
             * http://habrahabr.ru/post/216865/
             *
             * in_array($value, $list)
             */

            $found = false;

            foreach($list as $item)
            {
                if ($value == $item)
                {
                        $found = true;
                    break;
                }
            }

            if ($found == false)
            {
                $this->errors[] = $this->labels[$attribute] . ' не найден в списке возможных вариантов';
            }

            return $found;
        }



        /**
         * @param string $attribute поле для проверки
         * @param string $value содержимое поля для проверки
         * @param string $exts разрешенный расширения файлов, указывается через запятую
         * @return bool
         */

        public function validator_file_ext($attribute, $value, $exts)
        {
            if ($this->validator_in_list($attribute, pathinfo($value, PATHINFO_EXTENSION), explode(', ', $exts)) == false)
            {
                $this->errors[] = $this->labels[$attribute] . ' может быть одним из следующих форматов: '.$exts;
                return false;
            }
        }


        /**
         * @param string $attribute поле для проверки
         * @param string $value содержимое поля для проверки
         * @param array $size array('name', 'length', array('min' => 2, 'max' => 19');
         */

        public function validator_length($attribute, $value, $size)
        {
            if (isset($size['min']) && mb_strlen($value) < $size['min'])
            {
                $this->errors[] = $this->labels[$attribute] . ' должен содержать более '.$size['min'] . ' символов';
            }

            if (isset($size['max']) && mb_strlen($value) > $size['max'])
            {
                $this->errors[] = $this->labels[$attribute] . ' может содержать менее ' . $size['max'] . ' символов';
            }
        }





        /***
         * @param string $attribute поле для проверки
         * @param string $value содержимое поля для проверки
         * @return bool
         */

        public function validator_captcha($attribute, $value)
        {
            if (empty($_SESSION['captcha']) || trim(strtolower($value)) != $_SESSION['captcha'])
            {
                $this->errors[] = 'Неверный проверочный код';
                return false;
            }
        }



        /**
         * @param string $attribute поле для проверки
         * @param string $value содержимое поля для проверки
         * @param string $pattern регулярное выражение для проверки
         * @return bool
         */

        public function validator_match($attribute, $value, $pattern)
        {
            if (!preg_match($pattern, $value))
            {
                $this->errors[] =  $this->labels[$attribute] . ' не соответствует формату';
                return false;
            }

            //return preg_match($pattern, $value);
        }



        /**
         * @param string $attribute поле для проверки
         * @param string $url URL ля проверки
         * @return bool
         */

        public function validator_url($attribute, $url)
        {
            if(filter_var($url, FILTER_VALIDATE_URL, FILTER_FLAG_PATH_REQUIRED) == false)
            {
                $this->errors[] =  $this->labels[$attribute] . ' содержит не корректный URL';
                return false;
            }
        }



        /**
         * @param string $attribute поле для проверки
         * @param string $value содержимое поля для проверки
         * @return bool
         */

        public function validator_email($attribute, $value)
        {
            if (filter_var($value, FILTER_VALIDATE_EMAIL) == false)
            {
                $this->errors[] = 'Проверьте правильность ввода e-mail';
                return false;
            }
        }



        /**
         * @param string $attribute поле для проверки
         * @param string $value содержимое поля для проверки
         * @return bool
         */


        public function validator_login($attribute, $value)
        {

            if (
                preg_match("#^[a-zа-яё][a-zа-яё0-9\-\_\ ]{2,31}$#ui", $value) == false||
               (preg_match("#[a-z]+#ui", $value) && preg_match("#[а-яё]+#ui", $value)))
            {
                $this->errors[] = 'Проверьте правильность ввода логина';
                return false;
            }
        }


        /**
         * @param string $attribute поле для проверки
         * @param string $value содержимое поля для проверки
         * @param array $search массив с названиям таблицы и поля для проверки: array('login', 'in_table', array('table' => 'user', 'row' => 'nick');
         * @return bool
         */

        public function validator_in_table($attribute, $value, $search)
        {
            global $sql;
            if ($sql->query('SELECT COUNT(*) FROM `'.$search['table'].'` WHERE `'.$search['row'].'` = "'.$sql->esc($value).'"')->result() == 0)
            {
                $this->errors[] = $this->labels[$attribute] . ' не найден';
                return false;
            }
        }


        /**
         * @param string $attribute поле для проверки
         * @param string $value содержимое поля для проверки
         * @param mixed $user_method пользовательская функция для проверки
         */

        public function validator_user_method($attribute, $value, $user_method)
        {
            $result = call_user_func($user_method, $value);
            if (!empty($result))
            {
                $this->errors[] = $result;
            }
        }


        /**
         * @param string $attribute поле для проверки
         * @param string $value содержимое поля для проверки
         * @param mixed $default_value значение по умолчанию
         */

        public function validator_default($attribute, &$value, $default_value)
        {
            if (empty($value))
            {
                $value = $default_value;
            }
        }


        /**
         * @param string $attribute поле для проверки
         * @param string $value содержимое поля для проверки
         * @return bool
         */

        public function validator_token($attribute, $value)
        {
            if (!CSRF::check($value, $attribute))
            {
                $this->errors[] = 'Ошибка при отправке сообщения, попробуйте заново. Если ошибка повторяется, сообщите администратору';
                return false;
            }
        }
    }